package org.vishia.odt.readOdt;

import java.awt.image.BufferedImage;
import java.awt.image.WritableRaster;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.TreeMap;

import javax.imageio.ImageIO;

import org.vishia.genJavaOutClass.GenJavaOutClass;
import org.vishia.msgDispatch.LogMessageStream;
import org.vishia.util.Arguments;
import org.vishia.util.Debugutil;
import org.vishia.util.ExcUtil;
import org.vishia.util.FileFunctions;
import org.vishia.util.StringFunctions;
import org.vishia.util.StringFunctions_C;
import org.vishia.xmlReader.GenXmlCfgJavaData;
import org.vishia.xmlReader.XmlCfg;
import org.vishia.xmlReader.XmlDataNode;
import org.vishia.xmlReader.XmlJzCfgAnalyzer;
import org.vishia.xmlReader.XmlJzReader;


public class ReadOdt  extends TranslateOdtCommon {

  /**Version, history and license.
   * <ul>
   * <li>2024-12-13 Hartmut: enhancement for Linux usage: {@link #gathWrHeader(XmlDataNode, boolean)} now also writes the Linux shell script lines in included files.
   * <li>2024-05-29 Hartmut creation. Necessity to handle Libre Office writer content.  
   * </ul>
   *
   * 
   * <b>Copyright/Copyleft</b>:
   * For this source the LGPL Lesser General Public License,
   * published by the Free Software Foundation is valid.
   * It means:
   * <ol>
   * <li> You can use this source without any restriction for any desired purpose.
   * <li> You can redistribute copies of this source to everybody.
   * <li> Every user of this source, also the user of redistribute copies
   *    with or without payment, must accept this license for further using.
   * <li> But the LPGL ist not appropriate for a whole software product,
   *    if this source is only a part of them. It means, the user
   *    must publish this part of source,
   *    but don't need to publish the whole source of the own product.
   * <li> You can study and modify (improve) this source
   *    for own using or for redistribution, but you have to license the
   *    modified sources likewise under this LGPL Lesser General Public License.
   *    You mustn't delete this Copyright/Copyleft inscription in this source file.
   * </ol>
   * If you are intent to use this sources without publishing its usage, you can get
   * a second license subscribing a special contract with the author. 
   * 
   * @author Hartmut Schorrig = hartmut.schorrig@vishia.de
   * 
   * 
   */
  //@SuppressWarnings("hiding")
  public final static String version = "2025-10-10";

  
  public static class CmdArgs extends TranslateOdtCommon.CommonArgs {
      
    /**The asciidoc file to write. */
    File fAdoc;
    
    /**The common markup file to write. */
    File fOutZmL;
    
    File XXdirMarkup;
    
    boolean bDebug;
    
    boolean h1file;
    
    /**If true then call {@link GenXmlCfgJavaData} with the used existing XmlCfg. */
    File dirCreateCfgJavaData;
    
    /**If not null then write a new analyzed XmlCfg to this file, which is gotten from only one read #fIn . */
    File fWriteXmlStruct;
 
    /**Set with argument '-viewdiff:difftool.ext', then this command is executed if all is finished.*/
    String sCmdDifftool;
    
    /**Set with argument '-dirCmpn:path/to/dirCmp' only for the diff tool
     * 
     */
    File dirSavedZmL;
    
    /**If a www link starts with this, replace also the absolute link if an anchor is replaced.
     * 
     */
    String swwwRoot = "XXX";  // initialvalue not found in sRef
    
 
    /**Max length of a normal text line till ", " or alternatively till ", " or etc. or only till a space. */
    int adocLineLength = 105;
    
    /**Max length of a normal text line till ". " or \n. */
    int adocLineSentence = 140;
    
    /**Max. length of a line which can contain a long <:@....>*/
    int adocLineLengthFocus = 500;
    
    /**This array describes the commands with its help and stores the result to CmdArgs. More is not necessary. 
     * @since 2023-09-23 moved from {@link org.vishia.fbcl.readSmlk.GenSmlk2FBcl}, there was using the older concept with {@link org.vishia.mainCmd.MainCmd.Arguments}
     *   now proper to {@link Arguments}. This list is possible, alternatively to direct instantiating in ctor.
     * */
    Argument[] argList1 = {
          new Argument("-odt", ":<path/file.odt>  odt file for input", new SetArgument(){ 
          @Override public boolean setArgument(String val) throws FileNotFoundException { 
            return CmdArgs.this.arg_odt(val);
          }})
        , new Argument("-asciidoc", ":<path/file.txt>  file for output", new SetArgument(){ 
          @Override public boolean setArgument(String val) throws FileNotFoundException { 
            DirName dirName = getDirNameReplaceSrcCmpn(val);
            CmdArgs.this.fAdoc = new File(dirName.dir, dirName.name);  
            return CmdArgs.this.fAdoc.getParentFile().exists();
          }})
        , new Argument("-ZmL", ":<path/file.txt>  ", new SetArgument(){ 
          @Override public boolean setArgument(String val) throws FileNotFoundException { 
            DirName dirName = getDirNameReplaceSrcCmpn(val);
            CmdArgs.this.dirZml = dirName.dir.getAbsoluteFile();
            int posExt = dirName.name.indexOf('.');
            CmdArgs.this.sExtZmL = dirName.name.substring(posExt);
            // it is the start (first) file
            CmdArgs.this.fOutZmL = new File(CmdArgs.this.dirZml, dirName.name).getAbsoluteFile();  
            FileFunctions.mkDir(dirName.dir);
            return dirName.dir.exists();  
          }})
//        , new Argument("-o", ":<path/file.txt>  file for output", new SetArgument(){ 
//          @Override public boolean setArgument(String val) throws FileNotFoundException { 
//            int posWildcard = val.indexOf('*');
//            int posName = val.lastIndexOf('\\')+1;
//            int posName2 = val.lastIndexOf('/')+1;
//            if(posName2 > posName) { posName = posName2; }
//            int posDot = val.lastIndexOf('.');
//            CmdArgs.this.sNameMarkup = posWildcard == posName ? CmdArgs.this.sNameDoc : val.substring(posName, posDot);
//            CmdArgs.this.sExtOut = CmdArgs.this.sExtMarkup = val.substring(posDot);
//            CmdArgs.this.dirMarkup = new File(posName == 0 ? "." : val.substring(0, posName)).getAbsoluteFile();
//            CmdArgs.this.fOutZmL = new File(CmdArgs.this.dirMarkup, CmdArgs.this.sNameMarkup + CmdArgs.this.sExtMarkup).getAbsoluteFile();  
//            return CmdArgs.this.dirMarkup.exists();
//          }})
        , new Argument("-www", ":<wwwRoot>>  root for www access to replace operation link anchors", new SetArgument(){ 
          @Override public boolean setArgument(String val) throws FileNotFoundException { 
            CmdArgs.this.swwwRoot = val;  
            return true;
          }})
        , new Argument("-h1file", " If given writes an extra file per H1-level chapter", new SetArgument(){ 
          @Override public boolean setArgument(String val) throws FileNotFoundException { 
            CmdArgs.this.h1file = true;  
            return true;
          }})
        , new Argument("-genJavaData", ":<path>  Generates new versions of Java data in pkg path from -analyzeXmlStruct or from given XmlCfg", new SetArgument(){ 
          @Override public boolean setArgument(String val) throws IOException { 
            File dirJava = new File(val).getAbsoluteFile();
            if(!dirJava.exists() || !dirJava.isDirectory()) {
              return CmdArgs.this.errMsg("-genJavaData:%s ERROR not found as directory", dirJava );
            } else {
              CmdArgs.this.dirCreateCfgJavaData = dirJava;
              return true;
          }}})
        , new Argument("-analyzeXmlStruct", ":D:/path/to/xmlCfg.txt optional, first analyze the input xml data and generate a new XmlCfg text file", new SetArgument() {
          @Override public boolean setArgument(String val) throws FileNotFoundException { 
            CmdArgs.this.fWriteXmlStruct = new File(val) ;  
            File dirWriteXmlStruct = CmdArgs.this.fWriteXmlStruct.getParentFile();
            FileFunctions.mkDir(dirWriteXmlStruct);
            return dirWriteXmlStruct.exists();
          }})
        , new Argument("-debug", " set internal Flag for debugging", new SetArgument() {
          @Override public boolean setArgument(String val) throws FileNotFoundException { 
            CmdArgs.this.bDebug = true ;  
            return true;
          }})
        , new Argument("-difftool", ":viewdiff.bat path to the diff tool called after all is finished", new SetArgument() {
          @Override public boolean setArgument(String val) throws FileNotFoundException { 
            CmdArgs.this.sCmdDifftool = val ;  
            return true;
          }})
        , new Argument("-dirSavedZmL", ":path/to/cmp  used as second argument dir for diff tool only, the saved ZmL files", new SetArgument(){ 
          @Override public boolean setArgument(String val) throws FileNotFoundException { 
            CmdArgs.this.dirSavedZmL = getDirRepaceSrcCmpn(val);
            return CmdArgs.this.dirSavedZmL.exists();  
          }})
      };
    
    
    CmdArgs(){
      super();
      super.aboutInfo = "...Reader content from odg for FunctionBlockGrafic";
      super.helpInfo="obligate args: -o:... ...input";
      for(Argument arg: this.argListCommon) { addArg(arg); }
      for(Argument arg: this.argList1) { addArg(arg); }
    }
  

    /**Prepares the info for '-odt=path/to/File.odt'.
     * <ul>
     * <li>If the file name contains a '++' in its mid, the left part is used for {@link CommonArgs#sDirSrcCmpn}.
     *   and the right part is used for {@link CommonArgs#sNameDoc}.
     *   With this information a '++.../' in the argument -ZmL:path/++defaultCmpn/ZmL/*.ZmL is replaced.
     * <li>If the file does not contain a ++, the whole name of the odt file is written to {@link CommonArgs#sNameDoc}
     *   and {@link CommonArgs#sNameDoc} is set to null. 
     * </ul>   
     * @param val
     * @return
     */
    protected boolean arg_odt(String val) {
      this.fIn = new File(val).getAbsoluteFile();
      boolean bOk = this.fIn.exists();
      if(bOk) {
        this.dirIn = CmdArgs.this.fIn.getParentFile();
        String sName = CmdArgs.this.fIn.getName();         
        int posSep = sName.indexOf("++");                  
        int posDot = sName.lastIndexOf('.');
        if(posSep >0) {                                    // possible "DIR++/name.odt"
          super.sDirSrcCmpn = sName.substring(0, posSep);  // then save "DIR"
          super.sNameDoc = sName.substring(posSep+2, posDot);
        } else {
          super.sNameDoc = sName.substring(0, posDot);     // save "name" or "DIR/name" without ".odt" 
        }
      }
      return bOk;
    }
    
    
    @Override public boolean testConsistence ( Appendable msg ) {
      boolean bOk = true;
      try {
        if(false) { msg.append("-dirFBcl:path/to/outdir is obligate\n"); bOk = false; }
      } catch(IOException exc) {
        System.err.println("Fatal Error, msg as Appendable has IOException " + exc.getMessage());
      }
      if(!bOk) {
        super.showHelp(msg);
      }
      return bOk;
    }


    
  }
  
  final private CmdArgs cmdArgs;
  
  private XmlCfg xmlCfgOdt = new XmlCfg(true);
  
  private Map<String, StyleOdt> idxDirectToStyle = new TreeMap<>();
  
  private Map<String, StyleImg> idxImgDirectToStyle = new TreeMap<>();
  
  /**This index is filled initially with a few styles which are expected,
   * especially TextPg and TextCol, because of {@link StyleOdt#pgBreakBefore} etc. known by knowledge.
   * The index is supplemented by found style names without direct style, which are just indirect styles.
   * The properties of the indirect styles are not known here, because the styles.xml is not evaluated.
   */
  private Map<String, StyleOdt> idxIndirectStyle = new TreeMap<>();
  
  private Map<String, String> idxDirectOnlyStyle = new TreeMap<>();
  
  private Map<String, String> idxUsedStyles = new TreeMap<>();
  
  private Map<String, String> idxStyleToShort = new TreeMap<>();
  
  private Map<String, String> idxShortToStyle = new TreeMap<>();
  
  
  /**All found label references in Table of contents
   * with internal reference for header, VML-Label, title text and page.*/
  private Map<String, LabelRef> idxLabelRef = new TreeMap<>();
  
  private List<LabelRef> listLabelRef = new LinkedList<>();
  
  /**Counter for non found, automatic numbered labels. */
  private int ctLabel = 0;
  
  private XmlDataNode nodeDirectStyles;
  
  private XmlDataNode nodeText;
  
  /**Node text:h with the title of the document.  
   * Set in {@link #gatherAllHeading(XmlDataNode)}.
   */
  private List<XmlDataNode> listNdTitle = new LinkedList<>();
  
  private StyleOdt styleP;
  
  
  /**Stores a found BOOKMARK in 
   * <code><text:bookmark-ref ... text:ref-name="__Link_BOOKMARK"></code>
   * to write it in <#label, text ....> to compare with the other refs.
   */
  private String sRefBookmark;
  
  /**Stores a found text in 
   * <code><text:bookmark-ref text:reference-format="chapter" text:ref-name="...">text</text:bookmark-ref></code>
   * to write it in <#label, text ....>
   */
  private String sRefChapter;
  
  
  
  /**Flag that a included file is read.
   * Used for conditional {@link #writeLabelRef()}, not for included files..
   */
  private boolean bIsInclude = false;
  
  /**Set to true if the title is found in odt. 
   * All content before is not written to ZmL. The title is necessary.
   * The content before the title is the left empty page or such, automatically created,
   * not as part of ZmL source text.*/
  private boolean bTitleFound = false;
  
  private boolean bInCodeBlock;
  
  private boolean bInColumns;
  
  /**The last anchor which is replaced for operation arguments. */
  private String lastHlinkPathToReplaced = null;
  
  private String lastXref = null;
  
  /**If not null, then replace the first ' ' of the next output with this char. */
 // private char cReplaceNextSpaceCharWith;
  
  //int[] chapterNr = new int[6]; 
  
  /**Buffer able to see in debug for buffered writer for markup and asciidoc.*/
  private StringBuilder sb = new StringBuilder(4096), sba = new StringBuilder(4096);
  
  /**The output channel for markup and asciidoc*/
  
  IllegalArgumentException excTOCnotUpdated = new IllegalArgumentException("Table of contents not updated");
  
  private Writer wrFrame, wr, wra, wrRep;
  
  
  static String sIndent = "\n                ";
  

  
  /**main for this class, with given prepared arguments 
   * Does not catch unexpected exceptions and does not System.exit(...), use it to execute in a Java environment. 
   * @param args prepared cmd line arguments
   * @return 0 if all is ok
   * @throws IOException 
   * @throws Exception if unexpected.
   */
  public static int amain ( CmdArgs args) throws IOException {
    ReadOdt thiz = new ReadOdt(args);
    int error = thiz.execute();
    thiz.log.close();
    return error;
  }



  /**main gets the arguments as String, 
   * but does not catch unexpected exceptions and does not System.exit(...), use it to execute in a Java environment. 
   * @param sArgs
   * @return 0 if all is ok
   * @throws IOException 
   * @throws Exception if unexpected.
   */
  public static int smain ( String[] sArgs, Appendable logHelp, Appendable logError) throws IOException {
    CmdArgs args = new CmdArgs();
    if(sArgs.length ==0) {
      args.showHelp(logHelp);
      return(1);                // no arguments, help is shown.
    }
    if(  ! args.parseArgs(sArgs, logError)
      || ! args.testConsistence(logError)
      ) { 
      return(2);                    // argument error
    }
    //LogMessageStream log = new LogMessageStream(System.out);
    int exitCode = amain(args);
    if(exitCode !=0 ) { System.out.printf(" EXIT-code=%d", exitCode); }
    System.out.println();
    return exitCode;
  }



  /**main for UFBglConv, invoked from cmd line. 
   * Catch and report an unexpected exceptions via console error output, returns an exit code 
   * @param sArgs
   * @return 0 if all is ok
   */
  public static void main ( String[] sArgs) {
    try {
      int exitCode = smain(sArgs, System.out, System.err);
      System.exit(exitCode);
    } catch (Exception e) {
      System.err.println("Unexpected: " + e.getMessage());
      e.printStackTrace(System.err);
      System.exit(255);
    }
  }



  /**
   * Note: close of LogMessageStream is in {@link #amain(CmdArgs)} where this instance is constructed.
   * This closes also a given {@link TranslateOdtCommon.CommonArgs#fLog} because of 
   * call {@link LogMessageStream#LogMessageStream(java.io.OutputStream, java.io.OutputStream, Appendable, Appendable, boolean, Charset) }
   * with trut for argument closeOnClose.   * @param cmdArgs
   * @throws IllegalCharsetNameException
   * @throws UnsupportedCharsetException
   */
  @SuppressWarnings("resource") public ReadOdt(CmdArgs cmdArgs) throws IllegalCharsetNameException, UnsupportedCharsetException {
    super(new LogMessageStream(cmdArgs.fLog, null, System.out, System.err, true, Charset.defaultCharset()));
    this.cmdArgs = cmdArgs;
    this.idxStyleToShort.put("ccode-Java", "cJava");
    this.idxStyleToShort.put("Text_20_body", "pStd");
    this.idxStyleToShort.put("List1Left", "*li1l");
  }


  
  /**
   * @return
   * @throws IOException
   * @SuppressWarnings("resource"} because {@link #wra} and {@link #wrRep} are conditional opened here, 
   *   and also correct conditional closed in {@link #wrClose()}, but this is not detect by the Java compiler.  
   */
  @SuppressWarnings("resource") public int execute() throws IOException {
    showArguments();
    //
    int err = 6;
    this.bIsInclude = this.cmdArgs.sNameDoc.indexOf('+') >0;   // if the filename contains a '+' it is an included file.
    defineIndirectStyles();                                // default, styles.xml is not read
    this.xmlCfgOdt.readFromJar(ReadOdt.class, "xmlCfgOdtNonSemantic.txt", this.log);
    XmlCfg xmlCfgOdtAnalyzed = null;
    if(this.cmdArgs.fWriteXmlStruct !=null) {
      XmlJzCfgAnalyzer cfgAnalyzer = new XmlJzCfgAnalyzer();
      try {
        xmlCfgOdtAnalyzed = cfgAnalyzer.readXmlStructZip(this.cmdArgs.fIn, "content.xml");
        xmlCfgOdtAnalyzed.writeToText(this.cmdArgs.fWriteXmlStruct, this.log);
      } catch(Exception exc) {
        this.log.writeError("Exception analyzeXmlStruct", exc);
      }
    }
    if(this.cmdArgs.dirCreateCfgJavaData !=null) {
      GenJavaOutClass.CmdArgs genXmlJavaDataArgs = new GenJavaOutClass.CmdArgs();
      genXmlJavaDataArgs.dirJava = this.cmdArgs.dirCreateCfgJavaData;
      genXmlJavaDataArgs.sJavaClass = "XmlForOdt";
      genXmlJavaDataArgs.sJavaPkg = "org.vishia.odt.xmlData";
      GenXmlCfgJavaData genXmlJavaData = new GenXmlCfgJavaData(genXmlJavaDataArgs, this.log);
      XmlCfg xmlCfgForJavaData;
      if(this.cmdArgs.fWriteXmlStruct !=null) {
        xmlCfgForJavaData = new XmlCfg(true);
        xmlCfgForJavaData.readCfgFile(this.cmdArgs.fWriteXmlStruct, this.log);
      } else {
        xmlCfgForJavaData = this.xmlCfgOdt;
      }
      genXmlJavaData.exec(xmlCfgForJavaData);
    }
    if(this.cmdArgs.fIn !=null) {  //======================== read the given odt file <::callReadXml.>
      XmlDataNode data = new XmlDataNode(null, "root", null);  //<:@<new root>.>
      String error = readXml(data, this.cmdArgs.fIn);  //<:@<readXml>.>
      if(error !=null) {
        this.log.writeError("\nERROR reading xml file %s: %s", this.cmdArgs.fIn.getAbsolutePath(), error);
        err = 4;
      } else {
        writeBackupFile(this.cmdArgs.fOutZmL, this.cmdArgs.dirOutBack, this.cmdArgs.sExtOut, this.cmdArgs);
        this.wra = this.cmdArgs.fAdoc == null ? null :new OutputStreamWriter(new FileOutputStream(this.cmdArgs.fAdoc), "UTF-8");
        this.wr = new java.io.OutputStreamWriter(new FileOutputStream(this.cmdArgs.fOutZmL), "UTF-8");
        this.wrRep = this.cmdArgs.fReport == null ? null :new OutputStreamWriter(new FileOutputStream(this.cmdArgs.fReport), "UTF-8");
        //this.wr = new java.io.FileWriter(this.cmdArgs.fOut);
        //======>>>>                   ====================>> convert to ZmL
        try { 
          readOdgxmlWriteZmL(data);   //<:@<prc>.>
          wrClose();
          if(!this.bIsInclude) {
            writeLabelRef();                     //---------- writes the NAME.Labels.txt only for comprehensive files.
          }
          System.out.printf("\n*** finished %s @%s", this.cmdArgs.fOutZmL.getName(), this.cmdArgs.fOutZmL.getParent()  );
          err = 0;         //<:.callReadXml.>
        }
        catch(Exception exc) {
          CharSequence sExc;
          if(exc == this.excTOCnotUpdated) {
            sExc = "\nERROR: Table of contents is not updated, update all is missing, do it!";
          } else {
            sExc= org.vishia.util.ExcUtil.exceptionInfo("\nERROR ", exc, 0, 10);
          }
          this.log.append(sExc);
          // restore the back vml file:
          wrClose();
          if(this.cmdArgs.dirOutBack !=null) {
            String sNameBack = this.cmdArgs.sNameDoc;
            if(FileFunctions.getCanonicalPath(this.cmdArgs.dirOutBack).equals(FileFunctions.getCanonicalPath(this.cmdArgs.dirIn))) {
              sNameBack += ".back";
            }
            File fMarkupBack0 = new File(this.cmdArgs.dirOutBack, sNameBack + this.cmdArgs.sExtZmL);
            boolean bOkRename;
            bOkRename = this.cmdArgs.fOutZmL.delete()                   // delete the new created file
                      & fMarkupBack0.renameTo(this.cmdArgs.fOutZmL);    // and rename the last backup file with this number.
            if(bOkRename) { this.log.writef("\n%s successfull restored from: %s", this.cmdArgs.sNameDoc + this.cmdArgs.sExtZmL, fMarkupBack0.getAbsolutePath()); }
            else { this.log.writef("\nERROR restoring %s from %s", this.cmdArgs.sNameDoc + this.cmdArgs.sExtZmL, fMarkupBack0.getAbsolutePath());}
          }
          err = 3;
        }
        if(this.cmdArgs.sCmdDifftool !=null) {
          executeViewDiff();
        }
      }
    }
    this.log.close();
    return err;
  }

  
  
  private void showArguments () {
    System.out.printf("\n*** org.vishia.idt.readOdt.ReadOdt version %s", version);
    System.out.printf("\n  input=%s", this.cmdArgs.fIn );
    System.out.printf("\n  output=%s/*%s", this.cmdArgs.dirZml, this.cmdArgs.sExtOut );
    System.out.printf("\n  SrcCmpn=%s", this.cmdArgs.sDirSrcCmpn );
    System.out.printf("\n  currDir=%s", this.cmdArgs.sCurrDir );
    System.out.printf("\n  Backup-dir %s", this.cmdArgs.dirOutBack);
    System.out.printf("\n  -report=%s", this.cmdArgs.fReport);
    System.out.printf("\n  -dirDbg=%s", this.cmdArgs.dirDbg);
    System.out.printf("\n  Labels write to=%s/*%s", this.cmdArgs.dirLabel, this.cmdArgs.sNameLabel);
    System.out.printf("\n  www-root=%s", this.cmdArgs.swwwRoot);
    System.out.printf("\n  -wrelhtml=%s", this.cmdArgs.sRefBesideInWWWHtml);
    System.out.printf("\n  -wrelpdf=%s", this.cmdArgs.sRefBesideInWWWPdf);
    System.out.printf("\n  -alinkhtml=%s", this.cmdArgs.sRefBesideHtml);
    System.out.printf("\n  -alinkpdf=%s", this.cmdArgs.sRefBesidePdf);
    System.out.printf("\n  sExtRefBesideHtml=%s", this.cmdArgs.sExtRefBesideHtml);
    System.out.printf("\n  sExtRefBesidePdf=%s", this.cmdArgs.sExtRefBesidePdf);
    System.out.printf("\n  View Diff, compare after: %s", this.cmdArgs.sCmdDifftool);
    System.out.printf("\n  dir of saved ZmL=%s", this.cmdArgs.dirSavedZmL);
    if(this.cmdArgs.dirCreateCfgJavaData !=null) {
      System.out.printf("\n  -genJavaData=%s", this.cmdArgs.dirCreateCfgJavaData);
    }
    
  }
  
  
  /**Invokes the process in OS to show the Diff Tool adequate the option '-difftool:executable'*/
  private void executeViewDiff() {
    List<String> args = new LinkedList<>();
    if(this.cmdArgs.sCmdDifftool.endsWith(".bat")) {
      args.add("cmd.exe");                                 // special solution for windows necessary for calling a '.bat' file
      args.add("/C");
    }
    args.add(this.cmdArgs.sCmdDifftool);
    args.add(this.cmdArgs.dirZml.getAbsolutePath());
    args.add(this.cmdArgs.dirSavedZmL.getAbsolutePath());
    ProcessBuilder prcViewDiff = new ProcessBuilder(args);
    //ProcessBuilder prcViewDiff = new ProcessBuilder(this.cmdArgs.sCmdDifftool, this.cmdArgs.dirZml.getAbsolutePath(), "D:\\vishia\\LibreOffc\\TemplateZmLodt\\SOURCE.wrk\\src.back\\templateOdtZmlTechnDocu\\ZmL");
    try {
      System.out.println("\n== difftool: " + this.cmdArgs.sCmdDifftool);
      System.out.printf("\n  %s ",args.toString());
      
      prcViewDiff.start();
      System.out.println("\n== difftool should run: ");
    } catch (IOException e) {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    Debugutil.stop();
  }
  
  
  
  /**This operation defines the indirect styles which should be general known with its specific properties.
   * For example the "Heading 1" with internal name "Heading_20_1"  is knwon with page break. 
   * Then an additional page break information is not generated.
   * Same is for "TextPg", "TextCol". 
   * <br>
   * But for styles which are not found here, this is for example "Heading_20_2" etc. the information &lt;::pageBreak.> is generated in ZmL
   * and internally the style is stored as "NAMEPg", as also for "NAMECol".
   */
  private void defineIndirectStyles () {
    StyleOdt sQ = new StyleOdt('Q', "Quotation", null, 'i', '?', '?',false, false, 0);
    StyleOdt sE = new StyleOdt('E', "Emphasis", null, 'i', 'b', '?',false, false, 0);
    StyleOdt sS = new StyleOdt('S', "Strong_20_Emphasis", null, '?', 'b', '?',false, false, 0);
    this.idxIndirectStyle.put("Title", new StyleOdt('p', "Title", null, '?', '?', '?', true, false, 0));  // Title page should have page break before.
    this.idxIndirectStyle.put("TextPg", new StyleOdt('p', "TextPg", null, '?', '?', '?', true, false, 0));
    this.idxIndirectStyle.put("TextCol", new StyleOdt('c', "TextCol", null, '?', '?', '?', false, true, 0));
    this.idxIndirectStyle.put("ImgCaptionTextCol", new StyleOdt('i', "ImgCaptionTextCol", null, '?', '?', '?', false, true, 0));
    this.idxIndirectStyle.put("ImgCaptionTextPg", new StyleOdt('i', "ImgCaptionTextPg", null, '?', '?', '?', true, false, 0));
    this.idxIndirectStyle.put("ImgCaptionText", new StyleOdt('i', "ImgCaptionText", null, '?', '?', '?', false, false, 0));
    this.idxIndirectStyle.put("modif_i??", sQ);    //Modification style for fo:style = italic
    this.idxIndirectStyle.put("modif_?b?", sS);    //Modification style for fo:style = bold
    this.idxIndirectStyle.put("modif_ib?", sE);    //Modification style for fo:style = bold italic
    this.idxIndirectStyle.put("Q", sQ);    //Modification style for fo:style = italic
    this.idxIndirectStyle.put("S", sS);    //Modification style for fo:style = bold
    this.idxIndirectStyle.put("E", sE);    //Modification style for fo:style = bold italic
    this.idxIndirectStyle.put("modif_??1", new StyleOdt('1', "Subscript", null, '?', '?', '1',false, false, 0));    //Modification style for fo:style = italic
    this.idxIndirectStyle.put("modif_??2", new StyleOdt('2', "SuperScript", null, '?', '?', '2',false, false, 0));    //Modification style for fo:style = italic
    this.idxIndirectStyle.put("modif_i?1", new StyleOdt('3', "SubScriptItalic", null, 'i', '?', '1',false, false, 0));    //Modification style for fo:style = italic
    this.idxIndirectStyle.put("Heading_20_1", new StyleOdt('\0', "Heading_20_1", null, '?', '?', '?', true, true, 0));
    // The non existing indirect style for section with two columns.
    this.idxIndirectStyle.put("Column2", new StyleOdt('\0', "Column2", null, '?', '?', '?', false, false, 2));
  }
  
  
  /**Reads completely the content.xml from the given odt file <::readXml.>
   * and stores the data in the data instance.
   * @param data The root node for all XML data
   * @param fInOdt the odt file is a zip file
   * @return null or an error message
   * @throws IOException for file operation
   */
  private String readXml (XmlDataNode data, File fInOdt) throws IOException {
    String sFileOdt = fInOdt.getName();
    XmlJzReader xmlRd = new XmlJzReader();
    xmlRd.setNamespaceEntry("xml", "XML");  // it is missing
    xmlRd.setCfg(this.xmlCfgOdt);
    //xmlReader.setDebugStopTag("text:span");
    xmlRd.openXmlTestOut( new File(this.cmdArgs.dirDbg, sFileOdt + "-back.xml")); //fout1);
    String error = xmlRd.readZipXml(                                             //<:@<data>.>
               fInOdt, "content.xml", data );
    return error;
  } //<:.readXml.>
  
  
  
  /**This is the main routine after reading the whole XML file content.xml from odg.
   * @param ndRootOdg the root node of the content.xml
   * @return
   * @throws IOException
   */
  public String readOdgxmlWriteZmL(XmlDataNode ndRootOdg) throws IOException{
    XmlDataNode docu = ndRootOdg.allNodes.get(0);
    this.nodeDirectStyles = docu.singleNodes.get("office:automatic-styles");
    gatherDirectStyles(this.nodeDirectStyles);
    XmlDataNode nodeBody = docu.singleNodes.get("office:body");
    this.nodeText = nodeBody.singleNodes.get("office:text");
    gatherAllHeading(this.nodeText);
    writeTitle();
    writeAsciidocHead();
    writeZmlHead();
    gatherContent(this.nodeText, 0);
    return null;
  }
  
  
  /**Gather all direct styles in the document. It is only interesting for some dedicated properties
   * such as page break or column break which is added to a indirect style. 
   * The direct styles are stored in {@link #idxDirectOnlyStyle}.
   * The transfer from a direct style to the (indirect) parent style is stored in {@link #idxDirectToStyle} for non images
   * and {@link #idxImgDirectToStyle} for images.
   * @param nodeDirectStyles The direct style node.
   * @return
   */
  private String gatherDirectStyles(XmlDataNode nodeDirectStyles) {
    Iterable<XmlDataNode> styles = nodeDirectStyles.iterNodes("style:style");
    for(XmlDataNode ndStyle : styles) {
      String name = ndStyle.attribs.get("style:name");
//      if(name.equals("Column2"))
//        Debugutil.stop();
//      if(name.equals("P11"))
//        Debugutil.stop();
      String styleFamily = ndStyle.attribs.get("style:family");
      String styleParent = ndStyle.attribs.get("style:parent-style-name");
      String sBreak = "";
      String sFontStyle = null;
      String sFontWeight = null;
      String stylePosition = null;
      int nColumn = 0;
      XmlDataNode npProps = ndStyle.getFirstNode("style:paragraph-properties");
      XmlDataNode ntProps = ndStyle.getFirstNode("style:text-properties");
      if(ntProps !=null) {
        sFontStyle = ntProps.getAttrib("fo:font-style");   // italic
        if(sFontStyle !=null) {
          Debugutil.stop();
        }
        sFontWeight = ntProps.getAttrib("fo:font-weight"); // bold
        stylePosition = ntProps.attribs.get("style:text-position");
      }
      if(npProps !=null) {
        sBreak = npProps.getAttrib("fo:break-before");
        if(sFontStyle ==null) {
          sFontStyle = npProps.getAttrib("fo:font-style"); // https://www.data2type.de/xml-xslt-xslfo/xsl-fo/xslfo-referenz/attribute/font-style
        }                                                  //  says for XLS-FO: "italic", "oblique", "backslant", "normal"
        if(sFontWeight == null) {
          sFontWeight = npProps.getAttrib("fo:font-weight"); // https://www.data2type.de/xml-xslt-xslfo/xsl-fo/xslfo-referenz/attribute/font-weight
        }                                                  // "bold" = 700, "bolder" = 900, "lighter" = 200, "100", ... "900": "400" = normal , "700" = bold
      } else {
        npProps = ndStyle.getFirstNode("style:section-properties");
        if(npProps !=null) {
          XmlDataNode ndCol = npProps.getFirstNode("style:columns");
          if(ndCol !=null) {
            nColumn = getAttribNumber(ndCol, "fo:column-count", 1);
            if(styleParent == null && styleFamily.equals("section")) {
              styleParent = "Column" + nColumn;       // this is the real virtual existing column indirect style which is not exists in LibreOffc
            }
          }
        }
      }
      boolean bPgBreak = sBreak !=null && sBreak.equals("page");
      boolean bColBreak = sBreak !=null && sBreak.equals("column");
      char cItalic = sFontStyle !=null ? sFontStyle.charAt(0) : '?';   //'i' for italic
      char cPosition = '?';
      if(stylePosition !=null) {
        Debugutil.stop();
        if(stylePosition.startsWith("super")) { cPosition = '2'; }
        else if(stylePosition.startsWith("sub")) { cPosition = '1'; }
        else {
          this.log.writeInfo("Info: unknown position attribute: %s", stylePosition);
        }
      }
      char cBold = sFontWeight !=null ? sFontWeight.equals("bold") ? 'b' : sFontWeight.charAt(0) : '?';
      if(cItalic!='i' && cItalic!='n' && sFontStyle !=null) { this.log.writeWarning("\nfontstyle other then italic: %s", sFontStyle);}
      if(cBold!='b' && cBold!='n' && sFontWeight !=null) { this.log.writeWarning("\nfontweight other then bold: %s", sFontWeight);}
      //
      if(styleParent !=null && styleParent.equals(name)) {
        //The style name in odt is the same as existing indirect style, only for Column2 etc. relevant.  
      } else {
        if(styleFamily.equals("graphic")) {
          XmlDataNode ndGProps = ndStyle.getFirstNode("style:graphic-properties");
          String vPos, hPos, wrap, hRel, vRel;
          vPos = ndGProps.getAttrib("style:vertical-pos");
          hPos = ndGProps.getAttrib("style:horizontal-pos");
          wrap = ndGProps.getAttrib("style:wrap");
          hRel = ndGProps.getAttrib("style:horizontal-rel");
          vRel = ndGProps.getAttrib("style:vertical-rel");
          StyleImg styleImg = new StyleImg(name, styleParent, vPos, hPos, wrap, hRel, vRel);
          this.idxImgDirectToStyle.put(name, styleImg);
        } else {                                           // the original given direct style is stored here with the original given name
          StyleOdt styleOdt = new StyleOdt('\0', name, styleParent, cItalic, cBold, cPosition, bPgBreak, bColBreak, nColumn);
          this.idxDirectToStyle.put(name, styleOdt);       // the parent may be existing or not.
        }
      }
      if(styleParent == null) {  //-------------------------- It is a style nuance which is applicable to any style. For example 'italic'
        this.idxDirectOnlyStyle.put(name, styleFamily);    // yet only for info
      } 
      else {
        String styleFamilyUsed = this.idxUsedStyles.get(styleParent);
        if(styleFamilyUsed !=null && !styleFamilyUsed.equals(styleFamily)) {
          Debugutil.stop();
        }
        this.idxUsedStyles.put(styleParent, styleFamily);
      }
    }
    return null;
  }

  
  
  private void writeAsciidocHead () throws IOException {
    wrAdoc("\n"); wrAdoc(":toc: left");
    wrAdoc("\n"); wrAdoc(":toclevels: 5");
    wrAdoc("\n"); wrAdoc(":sectnums:");
    wrAdoc("\n"); wrAdoc(":sectlinks:");
    wrAdoc("\n"); wrAdoc(":max-width: 55em");
    wrAdoc("\n"); wrAdoc(":prewrap!:");
    wrAdoc("\n"); wrAdoc(":cpp: C++");
    wrAdoc("\n"); wrAdoc(":cp: C/++");
    wrAdoc("\n"); wrAdoc(":ldquo: \u201c");
    wrAdoc("\n"); wrAdoc(":rdquo: \u201d");
    wrAdoc("\n"); wrAdoc(":lgquo: „");
    wrAdoc("\n"); wrAdoc(":stylesheet: ./Asciidoc_StyleA/CppJava.css");
  }



  private void writeZmlHead () throws IOException {
    wrm("\n\n### ZmL Markup vishia-" + version);
    StringBuilder sbTextReplaceChars = new StringBuilder(WriteOdt.sTextReplace.length);
    for(String[] check: WriteOdt.sTextReplace) {           // only as comment, write some UTF character 
      char cc = check[0].charAt(0);                        // to identify this text file as UTF-8 coding by any used editor.
      if(cc >= 127) sbTextReplaceChars.append(cc);         // prevent write the newline! Write only higher codes.
    }
    wrm("\n### "); wrm(sbTextReplaceChars);  // to compare both
  }



  private String gatherAllHeading (XmlDataNode ndText) {
    boolean bFirstHeaderOrTitleDetected = false;
    StyleOdt style;
    for(XmlDataNode ndCheck : ndText.allNodes) {   //======__ iterate in order of nodes
      if( !bFirstHeaderOrTitleDetected && ndCheck.tag.equals("text:p")  //'Title' may be in a text:p, but before the first header
       && (style = getBaseStyle(ndCheck.getAttrib("text:style-name"), null)) !=null && style.name.equals("Title")  ){
        this.listNdTitle.add(ndCheck);
        bFirstHeaderOrTitleDetected = true;
      }
      if(ndCheck.tag.equals("text:h")) {    //--------------- text:h found
        //HHH
//        { String sHeaderText = ndCheck.getText();
//          if(sHeaderText.startsWith("Code generation for FBexpr")) {
//            Debugutil.stop();
//        } }
        int nOutline = getAttribNumber(ndCheck, "text:outline-level", 0);
        style = getBaseStyle(ndCheck.getAttrib("text:style-name"), null);
        if(style.name.equals("Title")) {
          this.listNdTitle.add(ndCheck);
          bFirstHeaderOrTitleDetected = true;
        } else if(nOutline >= 1 && nOutline <=6) {
          bFirstHeaderOrTitleDetected = true;
          gatherAllBookmarksInHeader(ndCheck);
        }
      } else if(ndCheck.tag.equals("text:section")) {
        gatherAllHeading(ndCheck);          //--------------- recursively call in text:section
      } else if(ndCheck.tag.equals("text:list")) {
        //XmlDataNode ndItem = ndCheck.getFirstNode("text:list-item");
        for(XmlDataNode ndItem : ndCheck.iterNodes("text:list-item")) {  
          // the text_list may contain more as one list-item on empty chapter.
          gatherAllHeading(ndItem);           //--------------- recursively call in text:list
        }
        for(XmlDataNode ndItem : ndCheck.iterNodes("text:list-header")) {  
          //------ this occurs if the header has removed its automatic number.
          gatherAllHeading(ndItem);           //--------------- recursively call in text:list
        }
      } else if(ndCheck.tag.equals("text:list")) {
        //------ this occurs if the header has removed its automatic number.
        gatherAllHeading(ndCheck);           //--------------- recursively call in text:list
      }
    }
    return null;
  }
  
  
  
  private void gatherAllBookmarksInHeader (XmlDataNode ndHead) {
    String sBookmarkTOC = null, sBookmarkRef = null;
    List<String> listRef = new LinkedList<>();
    boolean bBookmarkfound = false;
    for(XmlDataNode ndBookmark: ndHead.iterNodes("text:bookmark-start")) {
      bBookmarkfound = true;
      String sBookmark1 = ndBookmark.getAttrib("text:name");
      if(sBookmark1 == null) {
        // it is an error
      } else if( sBookmark1.startsWith("__RefHeading")) {
        sBookmarkTOC = sBookmark1;
      } else if( sBookmark1.startsWith("__RefNumPara")) {
        sBookmarkRef = sBookmark1;
      } else if( sBookmark1.startsWith("__Link_")) {         //SS_Bookmark
        listRef.add(sBookmark1.substring(7));
      }
    }
    if(bBookmarkfound) {
      LabelRef labelRef = null;
      if(sBookmarkTOC !=null) {
        labelRef = this.idxLabelRef.get(sBookmarkTOC);
      }
      if(labelRef == null && sBookmarkRef !=null) {
        labelRef = this.idxLabelRef.get(sBookmarkRef);
      }
      if(labelRef == null) {
        if(listRef.size() ==0) {
          listRef.add("$Label_" + (++this.ctLabel)); 
        }
        //for(String sBookmark: listRef) {
          newLabelRef(listRef, sBookmarkTOC, sBookmarkRef);
        //}
      } else {
        Debugutil.stop();
        assert(false);
      }
    } else {
      Debugutil.stop();
      this.log.writef("\nWARNING chapter title without bookmark: %s", ndHead.getText());
    }
  }
  
  
  
  private void newLabelRef (List<String> listBookmark, String sBookmarkTOC, String sBookmarkRef) {
    LabelRef labelRef = new LabelRef(sBookmarkTOC, sBookmarkRef, listBookmark, null, null);
    this.listLabelRef.add(labelRef);
    //------------------------------------------------------- store always the one labelRef with the three keys in index
    if(sBookmarkTOC !=null && this.idxLabelRef.get(sBookmarkTOC) == null) { 
      this.idxLabelRef.put(sBookmarkTOC, labelRef);
    }
    if(sBookmarkRef !=null && this.idxLabelRef.get(sBookmarkRef) == null) {
      this.idxLabelRef.put(sBookmarkRef, labelRef);
    }
    if(listBookmark !=null ) {//&& this.idxLabelRef.get(sBookmark) == null) {
      for(String sBookmark: listBookmark) {
        this.idxLabelRef.put(sBookmark, labelRef);
      }
    }
//    if(listBookmark !=null) { for(String sBookmark: listBookmark) { labelRef.idxLabel.put(sBookmark, sBookmark); } }
//    if(sBookmarkRef !=null) { labelRef.sRefNumPara = sBookmarkRef; }
//    if(sBookmarkTOC !=null) { labelRef.sRefHeading = sBookmarkTOC; }
    
  }
  
  
  
  
  /**The title should always be written as first. 
   * The document should only contain one paragraph formatted as title on top.
   * This assures same output for included files as also generate from an odt from an included file.
   * @param nodeContent
   * @return
   * @throws IOException
   */
  private String writeTitle ( ) throws IOException {
    wrm("../makeDocu/-LOffc-ZmL2odt.sh ./ "+ this.cmdArgs.sNameDoc + ".odt NOPAUSE;exit 0;\n");
    for(XmlDataNode ndTitle : this.listNdTitle) {  // get the first node
      wrAdoc("= ");  
      wrm("<::ZmL.>");  
      gathWrText(ndTitle, '\0');         // = Text of Title. Usual a simple text.
    }
    return null;
  }
  
  
  
  protected String gatherContent (XmlDataNode nodeContent, int deepnessSection) throws IOException {
    ListIterator<XmlDataNode> iterNodes = nodeContent.allNodes.listIterator();
    while(iterNodes.hasNext()) { //========================== while ...content
      XmlDataNode node = iterNodes.next();
//      if(node.tag.equals("$")) 
//        Debugutil.stop();
      String error = null;
      if(this.bInCodeBlock && ! node.tag.equals("text:p")) {   // text:p creates a second line in code
        wrAdoc("\n----\n"); wrm("\n<.Code>"); this.bInCodeBlock = false;
      }      
      if(node.tag.equals("text:h")) {
        error = gathWrHeader(node, false);
      }
      else if(node.tag.equals("text:list")) {
        error = gathWrList(node);
      }
      else if(node.tag.equals("text:p")) {
        char[] cStyleSpan = new char[1];
        StyleOdt styleP = getBaseStyle(node.attribs.get("text:style-name"), cStyleSpan);
        String sStyle = styleP.name;
        //if(this.bInCodeBlock) Debugutil.stopp();
        if(this.bInCodeBlock && !sStyle.startsWith("Code")) {
          wrAdoc("\n----\n"); wrm("\n<.Code>"); this.bInCodeBlock = false;
        }
        if(!this.bTitleFound) {   //======================= active text after title.
        }
        if( this.bTitleFound
         && !sStyle.startsWith("ExternRef")) {   //----------- Do nothing for p-style "ExternRef..."
          if(this.bInCodeBlock) {                //---------- Inside <:Code:...
            wrAdoc("\n"); wrm("\n");
          } else {
            wrAdoc("\n\n"); wrm("\n\n");
          }
          error = gathWrParagraph(node, iterNodes, 0, new String[] {"Text", "Text_20_body"});
        }
        else {                                   //---------- Text before title.
          this.bTitleFound |= getBaseStyle(sStyle, null).name.equals("Title");  // can be the text:p with style title.
        }
      }
      else if(node.tag.equals("table:table")) {
        error = gathWrTable(node);
      }
      else if(node.tag.equals("text:table-of-content")) {
        error = gathWrTableOfContent(node);
      }
      else if(node.tag.equals("text:section")) {
        // sStyleSect = node.attribs.get("text:style-name");
        StyleOdt styleSect = getBaseStyle(node.attribs.get("text:style-name"), null);
        String name = node.attribs.get("text:name");
        wrAdoc("\n\n//Section: "); wrAdoc(styleSect.name); //wr("]\n--");
        wrm("\n\n<::Section: "); wrm(styleSect.name); wrm(">");
        //wr(":"); wr(name); wr("--------------------$>");
        this.bInColumns = styleSect.name.startsWith("Column");
        error = gatherContent(node, deepnessSection +1);
        wrAdoc("\n\n//End-Section");
        wrm("\n\n<.Section>\n");
        if(deepnessSection >0) {
          wrAdoc(" " + deepnessSection);
        }
        this.bInColumns = false;
        wrAdoc("\n"); 
      }
      if(error !=null) {
        wrAdoc("\n//ERROR evaluating odt on node "); wrAdoc(node.tag); wrAdoc(": "); wrAdoc(error); wrAdoc("\n");
        wrm("\n//ERROR evaluating odt on node "); wrm(node.tag); wrm(": "); wrm(error); wrm("\n");
      }
    } // while ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ while ~~~~~~~~~~~~~~~~~~~~~~~~~
    if(this.bInCodeBlock ) {   // --------------------------- bInCodeBlock on end of document:
      wrAdoc("\n----\n"); wrm("\n<.Code>"); this.bInCodeBlock = false; // finish it
    }
    return null;
  }
  
  
  
  /**Writes a header as <pre>
   * '''
   * [#label]
   * ==== Chapter title
   * </pre>
   * @param ndHeader text:h or maybe text:p node
   * @param bInInclude false for the main file, true if write a file.ZmL to include after '__PART_' label ??? if read first time, true if read recursively to write the same header without the '__PART_' label in the include file.
   * @return
   * @throws IOException
   */
  private String gathWrHeader (XmlDataNode ndHeader, boolean bInInclude) throws IOException {
    String error = null;
    this.bTitleFound = true;                               // write content after the first chapter title also if the document title is not found.
    StyleOdt style = getBaseStyle(ndHeader.attribs.get("text:style-name"), null);
    int hLevel = getAttribNumber(ndHeader, "text:outline-level", 9);
    if(style.name.equals("Title")) {
      this.bTitleFound = true;
      hLevel = 0;    // Note: detected: Title has hLevel=1, but why? using Style Title is correct for this.
      return null;                // it is the title, already written.
    }
    if(hLevel ==0) {
      this.bTitleFound = true;
      return null;                // it is the title, already written.
    }
    List<String> listLabelRef = new LinkedList<>();
    String sIncludingName = null;
    LabelRef labelRef = null;                    //========== look for <text:bookmark-start ... Bookmark
    for(XmlDataNode ndLabel : ndHeader.iterNodes("text:bookmark-start")) {
      String sBookmark1 = ndLabel.getAttrib("text:name");
      if(sBookmark1 == null) {
        // it is an error
      } else {
        if( sBookmark1.startsWith("__Link_")) { sBookmark1 = sBookmark1.substring(7); }
        if(sBookmark1.startsWith("__PART_")) {
          int posBookmark2 = sBookmark1.indexOf('#');  //==== __PART_bookmark forces open an include file, but not on bInInclude call
          if(posBookmark2 >=0) {
            sIncludingName = sBookmark1.substring(7, posBookmark2);
          } else {
            sIncludingName = sBookmark1.substring(7);      // maybe start with 'RD_' for read only part
          }
          if(sIncludingName.startsWith("END_")) {          // flag for END of PART
            sIncludingName = "";                           // force close of the include file.
          }
        } else {                                 //========== any other Bookmark or __Link_ as Zml Label:
          LabelRef labelRef1 = this.idxLabelRef.get(sBookmark1);  // get the only one LabelRef for this chapter with one of these labels.
          if(labelRef1 !=null && labelRef1 != labelRef) {
            labelRef = labelRef1;
            for(String sLabel: labelRef.idxLabel.values()) {
              listLabelRef.add(sLabel);
          } }
          else if(labelRef1 == null) {  //-- labelRef not found, then anything in gatherAllHeading (...) is faulty:
            this.log.writef("\nWARNING problem with idxLabelRef: '%s'", sBookmark1);
            listLabelRef.add(sBookmark1);
          }
        }
      }
      //break; // One labelRef is sufficient, contains all, because gathered in gatherAllBookmarksInHeader (...)
    }
    //
    if(!bInInclude && this.wrFrame !=null && sIncludingName !=null) { //==== if wrFrame is given, and a new __PART_ starts, 
      if(this.wr !=null) {
        if(this.sb.length() > 0) {                           // complete all buffered stuff. But not on bInInclude 
          wrBufferToFile(this.sb, this.wr, null, true);
        }
        this.wr.close();
      }
      this.wr = this.wrFrame;    //========================== from now the original writer for header information is used only for the frame of the document.
      this.wrFrame = null;
    }
//    { String sHeaderText = node.getText();  //HHH
//      if(sHeaderText.startsWith("Code generation for FBexpr")) {
//        Debugutil.stop();
//    } }
    for(int ix=0; ix < 6-hLevel; ++ix) { wrAdoc("\n"); wrm("\n"); }   // write a distance to the chapter before, more for prior chapter level
    if(style.pgBreakBefore) {
      wrm("\n<::pageBreak.>\n"); wrAdoc("\n'''");
    }
    else if(style.colBreakBefore) {
      wrm("\n<::columnBreak.>\n"); wrAdoc("\n'''");
    }
    //String sBookmarkTOC = null, sBookmarkRef = null;
    if(hLevel <=6) {
//      if(!bInInclude) {                          //---------- count here the chapter number, it is not stored in the content.xml of odt
//        this.chapterNr[hLevel-1] +=1;            //|         
//        for(int ix = hLevel; ix < this.chapterNr.length; ++ix) {
//          this.chapterNr[ix] = 0;                //|
//        }                                        //|
//      }                                          //^---------
      wrAdoc("\n");
      for(String sBookmark : listLabelRef) {
        wrAdoc("[#"); wrAdoc(sBookmark); wrAdoc("]");
      }
      if(this.sb.length() > 0) {                           // complete all buffered stuff.
        wrBufferToFile(this.sb, this.wr, null, true);      // the buffer this.sb is empty now and will be filled with only the header info.
      }
      wrAdoc("\n=");  wrm("\n=");  //------------------------ === Title
      for(int ix = 0; ix<hLevel; ++ix) { wrAdoc("="); wrm("="); }
      wrAdoc(" "); wrm(" ");
      StringBuilder sbLogChapterNr = new StringBuilder();
//      for(int ix = 0; ix < hLevel; ++ix) {
//        sbLogChapterNr.append("" + this.chapterNr[ix]);
//        if(ix < hLevel-1) { sbLogChapterNr.append("."); }
//      }
//      wrm(sbLogChapterNr); wrm(" ");
      sbLogChapterNr.insert(0, " Chapter generated: ");
      for(String sBookmark : listLabelRef) {
        sbLogChapterNr.insert(0, "# " + sBookmark);
      }
      String text = ndHeader.getText();
      if(text !=null) {
        sbLogChapterNr.append(" ").append(text);
      }
      wrRep(sbLogChapterNr);
    } else {
      wrm("<:style:"); wrm(style.name); wrm(":Label:"); wrm(listLabelRef.get(0)); wrm(".>");
      wrAdoc("\n<$C:");             //--------------------------- <$C:style==bookmark>
      wrAdoc(style.name); wrAdoc("=="); wrAdoc(listLabelRef.get(0)); wrAdoc(">");
    }
    error = gathWrText(ndHeader, '\0');
    boolean bLabel = false;
    if(!bInInclude && sIncludingName !=null) {
      if( sIncludingName.length() >0) {
        wrm(" <:#__PART_"); wrm(sIncludingName);
      } else {
        wrm(" <:#__PART_END_");
      }
      bLabel = true;
    }
    for( String sBookmark : listLabelRef) {
      if(!bLabel) { wrm(" <:"); }
      bLabel = true;
      wrm("#"); wrm(sBookmark);
    }
    if(bLabel) { wrm(".>"); }
    //
    if(!bInInclude && sIncludingName !=null && sIncludingName.length() >0) {
      assert(listLabelRef.size() >0);
      String[] sHeader = new String[1];
      assert(this.sb.length() > 0);
      wrBufferToFile(this.sb, this.wr, sHeader, true);     // This prepares similar the header line for the H1 file.
      String sNameH1file = sIncludingName + this.cmdArgs.sExtZmL;
      //this.wr.append("\n<:@include:").append(sNameH1file).append(".>\n");
      //                                                   // instead this.wr is used for a new file per h1-chapter.
      File fInclOutZmL = null;
      try {
        if(sIncludingName.startsWith("RD_")) {   //---------- __PART_RD_ for a read only part
          this.wrFrame = this.wr;                          // same as for writing included part
          this.wr = null;                                  // but pause write in wrm(...)
        } else {
          fInclOutZmL = new File(this.cmdArgs.dirZml, sNameH1file);
          writeBackupFile(fInclOutZmL, this.cmdArgs.dirOutBack, this.cmdArgs.sExtOut, this.cmdArgs);
          Writer wrH1 = new java.io.OutputStreamWriter(new FileOutputStream(fInclOutZmL), "UTF-8");
          this.wrFrame = this.wr;    //====================== from now the original writer for header information is used only for the frame of the document.
          this.wr = wrH1;
          wrm("../makeDocu/-LOffc-ZmL2odt.sh "+ sIncludingName + " ;exit 0;\n");
          wrm("<::ZmL.>"); wrm(sIncludingName);            // Info on title page should be given.
          writeZmlHead();
          wrm("\n\n<::TOC-2.>Table of Contents");
          //wrm("\n\n<::pageBreak.>\n\n");
          gathWrHeader(ndHeader, true);  //======>>>>  ========== recursively call to write the same header but without __PART_Label in the included file.
        }
        //?? this.bIsInclude = false;                           // If the file includes another, it is not an included file.
      } catch(Exception exc) {
        CharSequence sError = ExcUtil.exceptionInfo("", exc, 0, 10);
        this.log.writef("Exception write include %s: %s", fInclOutZmL, sError);
      }                                    //NOTE: on exception it is furthermore written to this.wr, not changed.
    }
    
    return error;
  }
  

  
  
  
  /**Writes an paragraph from XML to VMU
   * It does not write a newline after the paragraph text because TODO
   * @param node
   * @param indent
   * @param stylesExpected
   * @return
   * @throws IOException
   */
  private String gathWrParagraph (XmlDataNode node, ListIterator<XmlDataNode> iterParentNodes, int indent, String[] stylesExpected) throws IOException {
    String error = null;
//    String text;
//    if( (text = node.getText()) !=null && text.startsWith("given download example"))
//      Debugutil.stop();
    String sStyle = node.attribs.get("text:style-name");
//    if(sStyle.equals("P18")) Debugutil.stopp();
    char[] cStyleSpan = new char[1];
    StyleOdt style = getBaseStyle(sStyle, cStyleSpan);
    if(style.name.startsWith("ExternRef")) {
      return null;       //---------------------------------- Do nothing for p-style "ExternRef..."
    }
    if(style.name.equals("CodeAsciidoc"))
      Debugutil.stop();
    if(!style.name.equals("TextPg") && style.pgBreakBefore) {
      wrm("\n<::pageBreak.>\n"); wrAdoc("\n'''");
    }
    else if(!style.name.equals("TextCol") && style.colBreakBefore) {
      wrm("\n<::columnBreak.>\n"); wrAdoc("\n'''");
    }
    boolean bImgCaption = style !=null && style.name.startsWith("ImgCaption");  //-------- The paragraph is only the caption to the image, 
    if(bImgCaption)
      Debugutil.stop();
    boolean bCodeInclude = style !=null && style.name.startsWith("CaptionCode");
    boolean bCode = style !=null && style.name.startsWith("Code");
    boolean bList = style !=null && style.name.startsWith("*");
    if(bCode) {
      Debugutil.stop();
    }
    if(bCodeInclude) {
      if(style.name.endsWith("Pg")) {
        wrm("\n<::pageBreak.>\n"); wrAdoc("\n'''");
      }
      else if(style.name.endsWith("Col")) {
        wrm("\n<::columnBreak.>\n"); wrAdoc("\n'''");
      }
      gathWrCodeInclude(node, iterParentNodes);
    }
    else if (style !=null && !bImgCaption                  // do not write the ImgCaption style because no text is created, only <:image:...>
      && (!bCode  || !style.equals(this.styleP))           // for bCode: do not write style again if it is unchanged. More paragraphs for code block.
       ) {                         //======================__ not a <:Code:.... or the end of a code block or the start of a code block
      boolean bStyleExpected = false;
      for(String styleCheck : stylesExpected) {
        if(style.name.equals(styleCheck)) {
          bStyleExpected = true;
          break;
        }
      }
      if(this.bInCodeBlock) {
        wrAdoc("----\n\n"); wrm("<.Code>\n\n");
        this.bInCodeBlock = false;
      }
      //wrAdoc("\n"); wrm("\n");
      if(bCode) {         // without "Code, only the STYLE after CodeSTYLE
        wrBufferToFile(this.sb, this.wr, null, true);      // write all stuff stored in File, because now direct writing.
        this.bInCodeBlock = true;
        wrAdoc("[Source, "); wrAdoc(style.name.substring(4)); wrAdoc("]\n----\n");
        wrm("<:Code:"); wrm(style.name.substring(4)); wrm(">\n");
      }
      else if(bList) {
        wrAdoc("* "); wrm("* ");
      }
      else if(!bStyleExpected) {
      //else if(!style.name.equals("Text")) { //-------------------- The style Text is the standard style for standard paragraphs
        wrAdoc("[."); wrAdoc(style.name); wrAdoc("]\n");
        wrm("<:p:"); wrm(style.name); wrm(".>");
      }
    }
    this.styleP = style;
    XmlDataNode nodeFrame = node.getFirstNode("draw:frame");
    if(nodeFrame !=null) {      //--------------------------- an image can also occur with a normal paragraph, non only with 'ImgCaption---' 
      //======>>>>
      gathImageInFrame(node, nodeFrame, bImgCaption);      
    }
    if(!bImgCaption && ! bCodeInclude) {
      error = gathWrText(node, cStyleSpan[0]);        // =====>>>>  -------- gathWrText()
    }
    return error;
  }
  

  
  
  private void gathImageInFrame ( XmlDataNode ndParagr, XmlDataNode nodeFrame, boolean bImgCaption) throws IOException {
    String styleOuterFrame = nodeFrame.getAttrib("draw:style-name");  // the style from the outer frame determines wrap
    XmlDataNode nodeTextbox = nodeFrame.getFirstNode("draw:text-box");
    if(nodeTextbox !=null) {
      XmlDataNode nodep = nodeTextbox.getFirstNode("text:p");
      if(nodep !=null) {
        String styleP = nodep.getAttrib("text:style-name");
        XmlDataNode ndFrameImg = nodep.singleNodes.get("draw:frame");
        if(ndFrameImg !=null) {
          gathWrFrameOrImage(nodep, ndFrameImg, styleOuterFrame, true);
        }
      }
    }
    XmlDataNode nodeImage = nodeFrame.singleNodes.get("draw:image");
    if(nodeImage !=null) {
      gathWrFrameOrImage(ndParagr, nodeFrame, styleOuterFrame, bImgCaption);
    }
  }
  

  
  private void gathWrCodeInclude ( XmlDataNode ndCaption, ListIterator<XmlDataNode> iterParentNodes) throws IOException {
    String text1 = getTextXml(ndCaption, true);             // may be the text has stupid span parts.
    //if(text1.contains("include:../../ExmplBandpassFilter/cmpGen/genSrcCmp/FBcl/ArrayBandpassFilter.fbd")) Debugutil.stopp();
    CharSequence text = replaceToBackslashSubscription(text1);
    //if(text != text1) Debugutil.stopp();   // stop if has changed the content
    final String sCode;
    if(!iterParentNodes.hasNext()) {             // the next node after CaptionCode contains the included code and also the code style.
      this.log.writef("\nError 'CaptionCode' content after missing: '%s'", text);
      sCode = "Cmd";                             // default: style CodeCmd
    } else {
      XmlDataNode ndNext = iterParentNodes.next();
      StyleOdt styleCode = getBaseStyle(ndNext.attribs.get("text:style-name"), null);
      if(styleCode !=null && styleCode.name.startsWith("Code")) {
        sCode = styleCode.name.substring(4);
        while(iterParentNodes.hasNext()) {  //===============__ skip all nodes which has the same Code style.
          ndNext = iterParentNodes.next();                   // it seems to be the included nodes.
          StyleOdt style = getBaseStyle(ndNext.attribs.get("text:style-name"), null);
          if(!style.name.equals(styleCode.name)) {
            break;                                           // another style is found.
          }
          for(XmlDataNode ndSpan : ndNext.iterNodes("text:span")) {
            StyleOdt styleSpan = getBaseStyle(ndSpan.attribs.get("text:style-name"), null);
            if(styleSpan.name.equals("cM")) {   //--------------- Marker in codeline found.
              String sMarker = ndSpan.getText();   // maybe compare the original code line and look whether the marker is also there.
              Debugutil.stop();                    // now to distinguish manual markers with markers in code? Here is a improve ability.
            }
          }
        }
        iterParentNodes.previous();
        //
      } else {
        iterParentNodes.previous();                        // do not skip this paragraph. It is not code.
        this.log.writef("\nError 'CaptionCode' content after should have style 'Code...': '%s'", text);
        sCode = "Cmd";                           // default: style CodeCmd especially if code not found.
      }
      wrAdoc("[Source, "); wrAdoc(sCode); wrAdoc("]\n----\n");
      wrBufferToFile(this.sb, this.wr, null, true);        // write all stuff stored in File, because now direct writing.
      //this.bInCodeBlock = true;
      wrm("<:Code:"); wrm(sCode); wrm(">\n");
      wrm("<::"); wrm(text); wrm(".>\n");                  // writes to ZmL with replaceToBackslashSubscription(...)
      wrAdoc(text); wrAdoc("[]\n");
      wrAdoc("\n----\n");
      wrm("<.Code>\n");
    }
    this.bInCodeBlock = false;
  }
  
  
  
  /**Data for {@link ReadOdt#gathWrText(XmlDataNode, char)}
   * and its inner operation.
   */
  private static class GathWrText {
    List<XmlDataNode> ndsSpan = null;
    StyleOdt styleSpan1 = null;
    
    char whatlast = 0;

    ListIterator<XmlDataNode> iterNodes;
    
    /**for the special construct '>>Link' whereby '>>' is also a link, the www link,
     * and 'Link' is the local link.  
     * It is possible that some more elements follow for link variants in odt, 
     * which are either used for the '@ref:#sRef' in ZmL, or skipped because no more necessary for ZmL.
     * If another element follows, then #bFollowLink is set to false.
     * 
     * */
    boolean bFollowLink = false;
    
    /**If {@link #bFollowLink} is set, if the local link is detected, and written to ZmL, then this flag is set.
     * If on end of links, this flag is not set, then the content of {@link #nodeLast} is written as link. 
     */
    boolean bLinkWritten = false;
    
    XmlDataNode nodeLast = null;
  }
  
  
  
  /**This is all inside a paragraph.
   * @param nodeP
   * @param cSpanParagr  If not 0 then the whole paragraph is marked with italic or bold, cStyleSpan is Q, E, S
   *   Then it writes non span parts as such span. 
   * @return
   * @throws IOException
   */
  private String gathWrText (XmlDataNode nodeP, char cSpanParagr) throws IOException {
    GathWrText thizi = new GathWrText();
    String error = null;
    if(nodeP.text !=null) {
      if(nodeP.text.contains("given download example")) Debugutil.stop();
      if(cSpanParagr !=0) {
        thizi.styleSpan1 = this.idxIndirectStyle.get("" + cSpanParagr);
        thizi.ndsSpan = new LinkedList<>();
        thizi.ndsSpan.add(nodeP);
        writeSpan(thizi.ndsSpan, thizi.styleSpan1);
        thizi.ndsSpan = null;
      } else {
        wrTextAdoc(nodeP.text);
        wrText(nodeP.text);
      }
    }
    //String spanContent = null;            // saver for span to prevent same span one after another dispersed
    thizi.iterNodes = nodeP.allNodes.listIterator();
    
     while(thizi.iterNodes.hasNext()) {
      XmlDataNode node = thizi.iterNodes.next();
      //if(node.text !=null && node.text.startsWith("<:p:List1")) Debugutil.stopp();
      error = gathWrTextNode(thizi, node, thizi.iterNodes, cSpanParagr);
    } // while iterNodes
    if(thizi.ndsSpan !=null) {                       // do not forget span on end.
      writeSpan(thizi.ndsSpan, thizi.styleSpan1);
    }
    return null;
  }


  
  /**Inner operation for one text node.
   * <ul>
   * <li>This operation calls itself for the {@link GathWrText#nodeLast} it it is set in specific conditions.
   * That's why it is extracted to handle one node.
   * </ul>
   * @param thizi the data class. Note: If this operation is part of the data class,
   * then the access to {@link ReadOdt} instance data are more complex. 
   * It is more simple and understandable to use this 'thizi' for specific data.
   * @param node The current node.
   * @param cSpanParagr
   * @return null or an error message.
   * @throws IOException
   */
  private String gathWrTextNode (GathWrText thizi, XmlDataNode node, ListIterator<XmlDataNode> iterNodes, char cSpanParagr) throws IOException {
    String error = null;
    if(node.tag.equals("text:span")) {         //---------- text:span range in text
      if(thizi.nodeLast !=null) {      //---------- nodeLast was ignored due to expected stuff, but it was not comming
        XmlDataNode nodeLast = thizi.nodeLast;
        thizi.nodeLast = null;                             // elsewhere recursive call may occur.
        gathWrTextNode(thizi, nodeLast, iterNodes, cSpanParagr);
      }
      String style0 = node.attribs.get("text:style-name");
      char[] cStyleSpan = null; //new char[1];
      StyleOdt styleSpan0 = getBaseStyle(style0, cStyleSpan);
      if(thizi.styleSpan1 !=null && styleSpan0!= thizi.styleSpan1) {  // the span style is changed
        writeSpan(thizi.ndsSpan, thizi.styleSpan1);             // then realize, write the span meaning and content.
        thizi.styleSpan1 = null; thizi.ndsSpan = null;
      }
      boolean bSpanDone = false;
      if(!bSpanDone) {
        String textSpan = getTextXml(node, true);        // Note: ignore more content in node, should not contain span in span
        if(textSpan !=null) {                            // ignore a ghost (empty) span
          if(textSpan.startsWith("<:style")) Debugutil.stop();
          if(styleSpan0 == null) {             //---------- it is a direct style span not evaluated
            assert(thizi.styleSpan1 == null);
            wrm(textSpan); wrAdoc(textSpan);             // ignore style, write the text.
            bSpanDone = true;
            textSpan = null;
          }
          else if(thizi.styleSpan1 == null) {        //---------- first node with this span style:
            thizi.styleSpan1 = styleSpan0;
            thizi.ndsSpan = new LinkedList<>();
            thizi.ndsSpan.add(node);
          } else { //if(style.equals(spanStyle) ) {    //---------- immediately following same span style and text, disadvantage of LOffc to do so
            thizi.ndsSpan.add(node);
      } } }
    } else {  // NOT "text:span"
      if(thizi.ndsSpan !=null) {         //--------------------vv another node than span, write out existing span.
        writeSpan(thizi.ndsSpan, thizi.styleSpan1);                  // from the detected span before with this style
        thizi.styleSpan1 = null; thizi.ndsSpan = null;
      }
      
      char whatnow = 0;
      if(node.tag.equals("text:a")) {
        boolean bXref = false;
        if(node.text !=null && node.text.equals(">>>") ) { // it is a link to a parallel document written as:
          bXref = checkWrCrossRefPdf(thizi, node, iterNodes); // >>>relLink.pdf: >>>relLink.html 
        }
        if(!bXref) {                             //--------vv not such a Xref link, check >>Javadoc or a normal link.
          gathWrTextNodeLink(thizi, node, iterNodes, cSpanParagr);
        }
      }
      else if(node.tag.equals("text:hidden-text")) {
        String sTextHidden = node.getAttrib("text:string-value");
        if(sTextHidden.startsWith("@ref#")) {
          if(!thizi.bLinkWritten) {
            int posInnerLabel = sTextHidden.startsWith("@ref#__Link_") ? 12: 5;
            wrm("<:@ref:#" + sTextHidden.substring(posInnerLabel) + ":.>");
          }
        } else {
          Debugutil.stop();  // not used hidden text ....
        }
      }
      else {
//        if(thizi.nodeLast !=null) {    //---------- nodeLast was ignored due to expected stuff, but it was not comming
//          XmlDataNode nodeLast = thizi.nodeLast;
//          thizi.nodeLast = null;
//          thizi.bFollowLink = false; thizi.bLinkWritten = false;
//          gathWrTextNode(thizi, nodeLast, cSpanParagr);
//        }
        if(node.tag.equals("$")) {
          //if(node.text.startsWith("<:p:List1")) Debugutil.stopp();
          if(cSpanParagr !=0) {
            thizi.styleSpan1 = this.idxIndirectStyle.get("" + cSpanParagr);
            thizi.ndsSpan = new LinkedList<>();
            thizi.ndsSpan.add(node);
            writeSpan(thizi.ndsSpan, thizi.styleSpan1);
          } else {
            wrTextAdoc(node.text);                                 // a free text in paragraph
            wrText(node.text);                                 // a free text in paragraph
          }
        }
        else if(node.tag.equals("text:p")) {
          Debugutil.stop();
        }
        else if(node.tag.equals("text:bookmark-ref")) {
          whatnow = gathWrXRef(node);
        }
        else if(node.tag.equals("text:line-break")) {
          if(this.bInCodeBlock) {
            wrAdoc("\n");
            wrm("\n");
          } else {
            wrAdoc(" <:nl>\n");
            wrm("\\n\n");
          }
        }
        else if(node.tag.equals("text:s")) {       // spaces
          if(thizi.whatlast != ' ') {        //ignore spaces after bookmark chapter nr 
            int n = getAttribNumber(node, "text:c", 1);   // Number of spaces
            while(--n >=0) { wrAdoc(" "); wrm(" ");}               // write spaces on <text:s...>
          }
        }
        else if(node.tag.equals("text:tab")) {       // spaces
          wrAdoc("\t"); wrm("\\t ");  
        }
        else if(node.tag.equals("text:section")) {
          Debugutil.stop();
          wrAdoc("\n//SECTION in p\n");
          wrm("\n<:SECTION in p>\n");
        }
      }
      thizi.whatlast = whatnow;
    } // NOT text:span
    return error;
  }
  

  
  /**This operation is called if a a:href with ">>>" as text is found. 
   * Supposed it is a link to a document in the same suite.
   * @param thizi
   * @param node1
   * @param iterNodes
   * @return false if it has not the expected structure.
   * @throws IOException
   */
  private boolean checkWrCrossRefPdf (GathWrText thizi, XmlDataNode node1, ListIterator<XmlDataNode> iterNodes) throws IOException {
    List<XmlDataNode> listLinkNodes = new LinkedList<>();
    String sStyle = node1.getAttrib("text:style-name");
    String sHref = node1.getAttrib("xlink:href");
    String sRef = null, sText = null;
    XmlDataNode node2;
    listLinkNodes.add(node1);
    String sLinkfileName = getLinkNameFromURL(sHref);
    while(thizi.iterNodes.hasNext()) {         //---------- gather all relevant nodes which are associated to ONE link:
      node2 = thizi.iterNodes.next();
      sStyle = node2.getAttrib("text:style-name");
      sHref = node2.getAttrib("xlink:href");              // BBB
      if( node2.tag.equals("text:a") 
         && getLinkNameFromURL(sHref).equals(sLinkfileName)
       || node2.tag.equals("text:hidden-text")
       || node2.tag.equals("$") && StringFunctions.indexAfterAnyChar(node2.text, 0, -1, ": ") == node2.text.length()
        ) {
        listLinkNodes.add(node2); 
      } else {
        thizi.iterNodes.hasPrevious();
        thizi.iterNodes.previous(); 
        break;
      }  
    }
    for(XmlDataNode node : listLinkNodes) {
      sStyle = node.getAttrib("text:style-name");
      sHref = node.getAttrib("xlink:href");
      int posInnerLabel;
      if(sStyle !=null && sStyle.equals("Reference") && (sHref == null || !sHref.contains(".pdf"))){
        sText = node.text;
      }
      if(sHref !=null && (posInnerLabel = sHref.indexOf('#') +1)>0) {
        if(sRef !=null) {
          if(!sRef.equals(sHref.substring(posInnerLabel))) {
            this.log.writef("\nERROR different inner labels: %s, %s", sRef, sHref);
          }
        } else {
          sRef = sHref.substring(posInnerLabel);
        }
      } else if(node.tag.equals("text:hidden-text")) {
        String sTextHidden = node.getAttrib("text:string-value");
        if(sTextHidden.startsWith("@ref#")) {
          posInnerLabel = sTextHidden.startsWith("@ref#__Link_") ? 12: 5;
          if(sRef !=null) {
            if(!sRef.equals(sTextHidden.substring(posInnerLabel))) {
              this.log.writef("\nERROR different hidden text inner labels: %s, %s", sRef, sHref);
            }
          } else {
            sRef = sTextHidden.substring(posInnerLabel);
          }
        } else {
          Debugutil.stop();  // not used hidden text ....
        }
      } else {
        // ignore links with posInnerLabel <=0 and text between. 
      }
    }
    if(sRef !=null) {
      if(sRef.startsWith("__Link_")) { sRef = sRef.substring(7); }
      if(sText == null) { 
        sText = "???"; 
      }
      wrm("<:@ref:#" + sRef + ":" + sText + ".>");
    } else {
      Debugutil.stop();   // maybe analyse listLinkNodes or repeat too see what's happen, this is a link with >>> but without the Xref.
    }
    return sRef !=null;
  }
  
  
  
  /**This is called for a &lt;text:a>... node, which is a hyperlink to outside.
   * If is checked, if the node has '>>' as text or "Reference" as style, then following nodes are gathered
   * which builds a combined link on odt, which is presented as one entry in ZmL.
   * <br>
   * If this condition is not true, the link are presented with {@link #gathWrXlink(XmlDataNode, ListIterator, boolean)}.
   * <br>
   * See {@link WriteOdt#writeChapterRefExtern(String, LabelRef)}
   * @param thizi
   * @param node1
   * @param cSpanParagr
   * @return
   * @throws IOException
   */
  private String gathWrTextNodeLink (GathWrText thizi, XmlDataNode node1, ListIterator<XmlDataNode> iterNodes, char cSpanParagr) throws IOException {
    String error = null;
    String sStyle = node1.getAttrib("text:style-name");
    String sHref = node1.getAttrib("xlink:href");
    //if(sHref.equals("https://vishia.org/fbg/pdf/Handling-OFB_VishiaDiagrams.pdf")) Debugutil.stopp();
    boolean bLinkWritten = false;
    String sRef = null, sText = null;
    XmlDataNode node2;
    if( node1.text !=null && node1.text.equals(">>")             //========== '>>SoftwareDocuLink' with '>>' contains the www link
      && iterNodes.hasNext() 
      && (node2 = iterNodes.next()) !=null                 // following node does not need to be a link
      && (sStyle = node2.getAttrib("text:style-name")) !=null
      && sStyle.startsWith("c")                            // following by a text style 'c...' (code style)
      ) {
      wrm(">><:"); wrm(sStyle); wrm(":"); 
      String text = node2.getTextAllSubNodes();
      if(text == null) { 
        text = "??"; 
      }
      wrText(text);
      wrm(".>"); // >><:STYLE:text.> no more.
      if(this.wra !=null) {
        error = gathWrXlink(node1, thizi.iterNodes, false);  // write link only for adoc
      }
      bLinkWritten = true;
    }  
    else if( sStyle !=null && sStyle.startsWith("c")              //========== a link node with style "c.." is also a reference to SoftwareDocuLink
        ) {
        wrm(">><:"); wrm(sStyle); wrm(":"); 
        String text = node1.getTextAllSubNodes();
        if(text ==null) { 
          text = "??"; 
        }
        wrText(text);
        wrm(".>"); // >><:STYLE:text.> no more.
        if(this.wra !=null) {
          error = gathWrXlink(node1, thizi.iterNodes, false);  // write link only for adoc
        }
        bLinkWritten = true;
      }  
    else {
      error = gathWrXlink(node1, thizi.iterNodes, true);  // write link as given here.
    }
    
      return error;
  }
  
  

  /**Gets the text inside a node, also regarding &lt;text:s...> and &lt;text:t for spaces and tabs
   * And also searching in sub nodes. Whereby the tag of sub nodes is ignored. This is usefull for &lt;text:span...
   * whereby span in span should ignored (it is only direct formatting).
   * <ul>
   * <li>Spaces are written in odt with &lt;text:s text:c="12"/> contains number of spaces (0x20)
   * <li>Tabs are written in odt with &lt;text:tab/> translated to 0x09
   * <li>Soft line breaks in a paragraph are written in odt with &lt;text:line-break/> translated to 0x0a
   * <li>Sub nodes are handled by recursively call of this operation, add their texts.
   * </ul>
   * It calls {@link #getTextXml(XmlDataNode, StringBuilder, boolean)} which does the work.
   * But if the node contains immediately text, this is returned.
   * @param nodep the given node
   * @param bSubnodes true then get the text also from all sub nodes, usefully for &lt;span...
   * @return
   */
  private String getTextXml (XmlDataNode nodep, boolean bSubnodes) {
    if(nodep.text !=null) return nodep.text;
    else {
      StringBuilder sbText = new StringBuilder();
      getTextXml(nodep, sbText, bSubnodes);
      return sbText.toString();
    }
  }
  
  
  
  /**Inner operation for Get text from a text XML node from odt: content.xml with special handling
   * called from {@link #getTextXml(XmlDataNode, boolean)} and recursively from itself.
   * @param nodep The text containing {@link XmlDataNode} node
   * @param sbText Here the text is added to, a {@link StringBuilder}
   * @param bSubnodes only on true read the text from sub nodes and insert it immediately.
   */
  private void getTextXml (XmlDataNode nodep, StringBuilder sbText, boolean bSubnodes) {
    if(nodep.allNodes == null || nodep.allNodes.size()==0) {
      String text = nodep.text;
      if(text !=null) { sbText.append(text); }
    } else {
      for(XmlDataNode node: nodep.allNodes) {
        if(node.tag.equals("text:s")) {                    // some spaces in XML
          String sNrSpaces = node.getAttrib("text:c");
          if(sNrSpaces == null) {
            sbText.append(' ');
          }
          else {
            int nrSpaces=StringFunctions_C.parseIntRadix(sNrSpaces, 0, 3, 10, null);
            while(nrSpaces >=10) {
              sbText.append("          ");
              nrSpaces -=10;
            }
            while(--nrSpaces >=0) {
              sbText.append(' ');
            }
          }
        } else if(node.tag.equals("text:tab")) {
          sbText.append('\t');
        } else if(node.tag.equals("text:line-break")) {
          sbText.append('\n');
        } else if(node.tag.equals("$")) {
          getTextXml(node, sbText, false);
        } else if(bSubnodes){
          getTextXml(node, sbText, bSubnodes);
        }
      }
    }
  }
  
  
  
  /**span is for any text part with specific formatting. 
     * Check whether it is <ul>
     * <li>Quotation or Citation, then write __text__
     * <li>Strong_20_Emphasis, then write **text**
     * <li>Emphasis, then write __**text**__
     * <li>ccode, then write `text`
     * <li>any other style, then write [STYLE]`text`
     * </ul>
     * The text it is checked whether it contains special strings which are interpreting by Asciidoc,
     * if yes then <ul>
     * <li>either `+text+` is written if '+' is not member of text,
     * <li>or `pass:[text]` is written if text does not contain ']'.
     * <li>or a combination is written with `+text+pass:[text2]` to prevent conflicts.
     * </ul>  
     * @param ndsSpan all nodes for this span entry
     * @param style The common style of maybe more as one span regions one after another, only the base style.
     * @return
     * @throws IOException
     */
    private String writeSpan ( List<XmlDataNode> ndsSpan, StyleOdt style) throws IOException {
      boolean bSpan = style != null;
      String sAdocEnd = "";
      if(!bSpan) {
        String textSpanAll = "";
        for(XmlDataNode node: ndsSpan) {
          textSpanAll += getTextXml(node, true);
        }
        wrm(textSpanAll); wrAdoc(textSpanAll);                   // without style, write only the text.
      }
      else {
        String textSpanTest = "";
        for(XmlDataNode ndSpan: ndsSpan) { textSpanTest += getTextXml(ndSpan,true); }
//        if(textSpanTest.contains("Textual")) Debugutil.stopp();
        //
//        if(node.text !=null && node.text.startsWith("ontent"))
//          Debugutil.stop();
//        if(textSpanAll !=null && textSpanAll.startsWith(" "))
//          Debugutil.stop();
        switch(style.cShort) {
        case 'Q': wrAdoc("__"); sAdocEnd = "__"; break;
        case 'E': wrAdoc("__**"); sAdocEnd = "**__"; break;
        case 'S': wrAdoc("**"); sAdocEnd = "**"; break;
        case '1': wrAdoc("~"); sAdocEnd = "~"; break;
        case '2': wrAdoc("^"); sAdocEnd = "^"; break;
        case '3': wrAdoc("__~"); sAdocEnd = "*__"; break;
        case '4': wrAdoc("__^"); sAdocEnd = "^__"; break;
        default: sAdocEnd = "";
        }
        if(sAdocEnd.length()>0) {
          wrm("<:" + style.cShort + ":");
        }
        else if(style.name.equals("Reference")) {   //------- Reference style is used only for bookmark references,
          String textSpanAll = "";
          for(XmlDataNode node: ndsSpan) {
            textSpanAll += getTextXml(node, true);
          }
          int zTxt = textSpanAll.length();                    // this is already processed,
          if(textSpanAll.startsWith("XREF:")) {        //------- or for the XREF:label designation
            String sXref = textSpanAll.substring(5);
            if(sXref.startsWith("#")) { sXref = sXref.substring(1); }
            if(sXref.equals("Impl-ReadOdg-PageShape-Pins")) Debugutil.stopp();
            wrAdoc("link:"); wrAdoc(sXref); wrAdoc(".todo.XREF[]");
            wrm("<:@ref:#" + sXref + ":.>");
            bSpan = false;
          } else {                                  //------- or for page number, if a digit is given.
            int[] parsedChars = new int[1];
            if( StringFunctions_C.parseIntRadix(textSpanAll, 0, zTxt, 10, parsedChars) !=0
             && parsedChars[0] == zTxt) {    //---------------- it is obviously a page number.
              wrAdoc("PDF@"); wrAdoc(textSpanAll);
              wrm("<:@page.>");
              bSpan = false;
            } 
            else {
              for(XmlDataNode ndSpan: ndsSpan) {
                for(XmlDataNode ndRef: ndSpan.iterNodes("text:bookmark-ref")) {
                  gathWrXRef(ndRef);           // Bookmark references should not use specific other character styles than "Reference"
                  bSpan = false;                           // do not write the span text
                }
              }
              if(bSpan) {
                wrm("<:Reference:");             //---------- Reference style without found xref 
                wrAdoc("__"); sAdocEnd = "__";             // then write the text with style 'Reference'
              }
            }
          }
        }
        else if(style.name.startsWith("modif_")) {              // a direct style with modifications detected, generate the Letter for the indirect style.
          if(style.cItalic =='i') {                        // ignore cItalic == 'n' for style = normal 
            //assert(false);
            wrAdoc("__"); sAdocEnd = "__";
            if(style.cWeight !=0) { wrAdoc("**"); sAdocEnd = "**__"; }
            String sStyle = style.cWeight ==0 ? "Q" : "E";  // complete with a table!
            wrm("<:" + sStyle + ":"); 
          } 
          else if(style.cWeight !='?') {
            //assert(false);
            assert(style.cItalic !='i' || style.cItalic == 'n');
            wrAdoc("**"); sAdocEnd = "**";
            wrm("<:S:"); //wrm("<:" + style.cWeight + ":"); 
          }
          else {                                           // the modif_n for cItalic == 'n' does not force a span
            bSpan = false;  
          }
        }
        else {
          String sStyle = style.name;
          String sStyleStart = null;
          for(String[] check: WriteOdt.sTextCtrlCommon) {
            if(sStyle.equals(check[4])) {
              sStyleStart = check[0];
              break;
            }
          }
          if(sStyleStart == null) {
            sStyleStart = "<:" + sStyle + ':';
          }
          boolean bCode = style !=null && style.name.equals("ccode");    
          if(bCode) { sStyle = "c"; }
          wrm(sStyleStart);                                // VML output only one character "<:Q:"
          boolean bItalic = style !=null && (style.name.equals("Quotation") || style.name.equals("Citation"));
          boolean bBold = style !=null && style.name.equals("Strong_20_Emphasis");
          boolean bItBold = style !=null && style.name.equals("Emphasis");
          if(bItalic) { wrAdoc("__"); sAdocEnd = "__"; }   // It is for Asciidoc
          else if(bBold) { wrAdoc("**"); sAdocEnd = "**"; }
          else if(bItBold) { wrAdoc("__**"); sAdocEnd = "**__";  }
          else if(bCode) { wrAdoc("`"); sAdocEnd = "`";  }
          else { 
            wrAdoc("["); wrAdoc(style.name); wrAdoc("]`"); sAdocEnd = "`"; //Asciidoc output
          }
        } // if kind of span
      } // if span
      if(bSpan) {
        for(XmlDataNode ndSpan: ndsSpan) {
          if(ndSpan.tag.equals("text:bookmark-ref")) {
            gathWrXRef(ndSpan);           // Bookmark references should not use specific other character styles than "Reference"
          } else {
            String textSpan = getTextXml(ndSpan,true);
            if(textSpan.equals("Value cast of input:"))
              Debugutil.stop();
            wrTextAdoc(textSpan);
            wrText(textSpan);
          }
        }
        wrAdoc(sAdocEnd); wrm(".>");
      }
      return null;
    }



  private String gathWrList (XmlDataNode nodeList) throws IOException {
    XmlDataNode nodeItem = nodeList.singleNodes.get("text:list-header");
    if(nodeItem !=null) {  //-------------------------------- exact one list item, it is for a numbering paragraph.
      gathWrListItemOrNumbHeader(nodeItem);
    } 
    else {
      if( (nodeItem = nodeList.singleNodes.get("text:list-item")) !=null) {
        gathWrListItemOrNumbHeader(nodeItem);
      } else {
        List<XmlDataNode> listItem = nodeList.multiNodes.get("text:list-item");
        if(listItem !=null) {
          for(XmlDataNode nodeItem1 : listItem) {
            gathWrListItemOrNumbHeader(nodeItem1);
          }
        }
        Debugutil.stop();
      }
    }
    return null;
  }
  
    
    
  /**Gather an item in a list, not known what is it.
   * It may be a numbered header, an image, or a really list item. 
   * @param nodeItem
   * @return
   * @throws IOException
   */
  private String gathWrListItemOrNumbHeader (XmlDataNode nodeItem) throws IOException {
    XmlDataNode nodeP = nodeItem.singleNodes.get("text:h");
    if(nodeP == null) {nodeP = nodeItem.singleNodes.get("text:p");}
    if(nodeP !=null) {  //--------------------------------- exact one node in text:list-item found, it may be exact one numbering paragraph
      String sOutlineLevel = nodeP.getAttrib("text:outline-level");
      if(sOutlineLevel !=null && sOutlineLevel.charAt(0) <= '6') {
        gathWrHeader(nodeP, false);
      } else {
        gathWrListItem(nodeP);
      }
    }
    XmlDataNode nodeListNested = nodeItem.getFirstNode("text:list");
    if(nodeListNested !=null) { //--------------------- possible nested lists if more as one chapter header are immediately one after another.
      gathWrList(nodeListNested);
    }
    return null;
  }
  
  
  private String gathWrListItem (XmlDataNode ndItem) throws IOException {
    XmlDataNode nodeFrame = ndItem.singleNodes.get("draw:frame");
    if(nodeFrame !=null) {
      String styleFrame = nodeFrame.getAttrib("draw:style-name");  // the style from the outer frame determines wrap
      return gathWrFrameOrImage(ndItem, nodeFrame, styleFrame, true);
    }
    else {
      StyleOdt style = getBaseStyle(ndItem.getAttrib("text:style-name"), null);
      wrAdoc("\n\n* ");
      wrm("\n\n* ");
      return gathWrText(ndItem, '\0');
    }
  }
  
  
  private String gathWrTable (XmlDataNode ndTable) throws IOException {
    String error = null;
    XmlDataNode ndTclmn = ndTable.getFirstNode("table:table-column");
    String sColumn = ndTclmn.getAttrib("table:number-columns-repeated");
    wrm("\n\n<::table:col="); wrm(sColumn); wrm(".>");
    for(XmlDataNode ndRow: ndTable.iterNodes("table:table-row")) {
      wrm("\n<:tr.>");
      for(XmlDataNode ndCell: ndRow.iterNodes("table:table-cell")) {
        wrm("\n  <:td");
        String spanned = ndCell.getAttrib("table:number-columns-spanned");
        if(spanned !=null) { wrm(":spanned="); wrm(spanned); }
        wrm(".>");
        ListIterator<XmlDataNode> iterNodes = ndCell.iterListNodes("text:p");
        while(iterNodes.hasNext()) {
          XmlDataNode ndP = iterNodes.next();
          gathWrParagraph(ndP, iterNodes, 4, new String[] { "Table_20_Contents" });
        }
      }
    }
    wrm("\n<.table>");
    return error;
  }
  
  
  
  private String gathWrTableOfContent (XmlDataNode node) throws IOException {
    XmlDataNode nd2 = node.getFirstNode("text:table-of-content-source");
    XmlDataNode ndIdxBody = node.getFirstNode("text:index-body");
    boolean bPgbreakBefore = false;
    char[] cStyleSpan = new char[1];
    String sTableTitle = "Table of contents";
    if(ndIdxBody !=null) {
      XmlDataNode ndIdxTitle = ndIdxBody.getFirstNode("text:index-title");
      if(ndIdxTitle !=null) {
        XmlDataNode ndp = ndIdxTitle.getFirstNode("text:p");
        if(ndp !=null) {
          StyleOdt styleTitle = getBaseStyle(ndp.getAttrib("text:style-name"), cStyleSpan);
          bPgbreakBefore = styleTitle !=null && styleTitle.pgBreakBefore;
          sTableTitle = ndp.getText();
      } }
      boolean bHasEntries = false;
      for(XmlDataNode ndEntry: ndIdxBody.iterNodes("text:p")) {
        XmlDataNode ndRef = ndEntry.getFirstNode("text:a");
        if(ndRef !=null) {
          bHasEntries = true;  
          String sHref = ndRef.getAttrib("xlink:href");
          if(sHref.startsWith("#")) {
            sHref = sHref.substring(1);
          }
          String sPage = null;
          String sTitle = getTextXml(ndRef, false);
          int posTab = sTitle.indexOf('\t');
          if(posTab >0) {
            sPage = sTitle.substring(posTab +1);
            sTitle = sTitle.substring(0, posTab);
          }
          LabelRef labelRef = this.idxLabelRef.get(sHref);
          if(labelRef == null) {
            List<String> listHref = new LinkedList<>();
            listHref.add(sHref);
            this.idxLabelRef.put(sHref, new LabelRef("?", "?", listHref, sTitle, sPage));
          } else {
            labelRef.sTitle = sTitle;
            labelRef.sPage = sPage;
          }
        }
      }
      if(!bHasEntries) {
        throw this.excTOCnotUpdated;
      }
    }
    if(nd2 !=null) {
      int nTillLevel = getAttribNumber(nd2, "text:outline-level", 0);
      String kind = nd2.getAttrib("text:index-scope");
      wrAdoc("\n");
      if(bPgbreakBefore) {
        wrAdoc("\n\n\n'''");
      }
      wrAdoc("\n//TOC"); 
      if(kind !=null) { wrAdoc("-"); wrAdoc(kind); }
      wrAdoc("-" + nTillLevel);
      wrAdoc(":");
      wrAdoc(sTableTitle);
      wrm("\n");
      if(bPgbreakBefore) {
        wrm("\n\n\n<::pageBreak.>");
      }
      wrm("\n<::TOC"); 
      if(kind !=null) { wrm("-"); wrm(kind); }
      wrm("-" + nTillLevel);
      wrm(".>");
      wrm(sTableTitle);
      return null;
    }
    else return "odt-MISSING <text:table-of-content-source...>"; 
  }



  private String gathWrFrameOrImage(XmlDataNode ndParagr, XmlDataNode ndFrame
  , String styleFrameRaw
  , boolean bCaption
  ) throws IOException {
    //String sCaption = ndParagr.getText();
    final StringBuilder sbCaption;
    if(!bCaption) {
      sbCaption = null;
    } else {
      sbCaption = new StringBuilder();
      for(XmlDataNode node: ndParagr.allNodes) {
        if(node.tag.equals("$")) {
          sbCaption.append(node.text);                       
        }
        else if(node.tag.equals("text:span")) {              // text:span range in caption text, ignore style
          sbCaption.append(node.text);                       
        }
      }
//      if(sbCaption.toString().contains("Complex_ifFB.png"))
//      Debugutil.stop();
      replaceToBackslashSubscription(sbCaption);              // replace in sbCaption immediately, it is a StringBuilder.
    }
    XmlDataNode ndBookmark = ndParagr.getFirstNode("text:bookmark-start");
    String sBookmark = null;
    if(ndBookmark != null) {
      sBookmark = ndBookmark.getAttrib("text:name"); //getBookmarkShort(ndBookmark, "text:name");
      if(sBookmark == null) {
        sBookmark = ndBookmark.getAttrib("draw:name"); //getBookmarkShort(ndBookmark, "draw:name");
      }
    }
    XmlDataNode nodeImage = ndFrame.singleNodes.get("draw:image");
    if(nodeImage !=null) {
      String sLink = nodeImage.getAttrib("xlink:href");
      StyleImg styleImg = this.idxImgDirectToStyle.get(styleFrameRaw);
      String sSep = " :: "; String sSepAdoc = "";
      if(sLink !=null) {
        if(sLink.startsWith("../")) {         //------------- relative link has in content.xml one more ../ as from file.odg 
          sLink = sLink.substring(1);                      // one parent level lesser, but write "./path..."
        }
        wrAdoc("\n\nimage::"); wrAdoc(sLink); wrAdoc("["); 
        wrm("\n<:@image:"); wrm(sLink);       // Hint: The image is always part of a paragraph. Write only one newline.
        if(sBookmark !=null) { 
          wrAdoc("id="); wrAdoc(sBookmark.substring(1));
          wrm(sSep); wrm("id="); wrm(sBookmark); 
          sSep = " :: "; sSepAdoc = ", ";
        }
        if(sbCaption !=null && sbCaption.length()>0) { 
          wrAdoc(sSep); wrAdoc("title="); wrAdoc(sbCaption); 
          wrm(sSep); wrm("title="); wrm(sbCaption); 
//          if(sbCaption.toString().contains("View 100%"))
//            Debugutil.stop();
          sSep = " :: "; sSepAdoc = ", ";
        }
        String style = "ImageCenter";
        if(styleImg !=null) { //depends on styleFrame
          if(styleImg.parentName.startsWith("Image")) {
            style = styleImg.parentName;                     // The image style is one of the Img... to favor use, it is satisfied.
          } else {
            String styleFloat =                              // get properties from the direct style properties
                styleImg.wrap == null ? ""          
                : styleImg.wrap.equals("left") ? "Floatleft" 
                : styleImg.wrap.equals("right") ? "Floatright"
                : "";
            style = "Image" + styleFloat;
          }
          wrm(sSep); wrm("style="); wrm(style); sSep = " :: ";
        } 
        int nFloat = style.indexOf("Float");
        if(nFloat >0) {
          String sFloat = style.substring(nFloat+5);
          wrm(sSep); wrm("float=" + sFloat); sSep = " :: ";
          wrAdoc(sSepAdoc); wrAdoc("float=" + sFloat); sSepAdoc = ", ";
          
        }
        String sRelX = ndFrame.getAttrib("style:rel-width");
        String sRelY = ndFrame.getAttrib("style:rel-height");
        if(sRelX !=null) {
          wrm(sSep); wrm("sRelX = " + sRelX); sSep = " :: ";
        }
        if(sRelY !=null) {
          wrm(sSep); wrm("sRelY = " + sRelY); sSep = " :: ";
        }
        String sXcm = ndFrame.getAttrib("svg:width");
        String sYcm = ndFrame.getAttrib("svg:height");
        int[] posMeas = new int[1];
        final float nX = StringFunctions_C.parseFloat(sXcm, 0, -1, posMeas);
        final float nY = StringFunctions_C.parseFloat(sYcm, 0, -1, posMeas);
        final float nXinch, nYinch;
        String measUnit = sYcm.substring(posMeas[0]);
        final float nRound;
        if(measUnit.equals("in")) { nXinch = nX; nYinch = nY; nRound = 100.0f; }
        else if(measUnit.equals("cm")) { nXinch = nX / 2.54f; nYinch = nY / 2.54f; nRound = 50.0f; }
        else if(measUnit.equals("mm")) { nXinch = nX / 25.4f; nYinch = nY /25.4f; nRound = 5.0f; }
        else if(measUnit.equals("pt")) { nXinch = nX / 72; nYinch = nY /72; nRound = 1.0f; }  // 1 pt = 0.3527778 mm
        else { nXinch = nX; nYinch = nY; nRound = 100.0f; this.log.writeWarning("unknown measurement for image: %s: %s\n", sLink, sYcm);}
        final float nXr = ((int)(nRound * nX + 0.5f)) / nRound;                 // round to on decimal after.
        final float nYr = ((int)(nRound * nY + 0.5f)) / nRound;
        //wrm(sSep); wrm("size="); wrm(sXcm); wrm("*"); wrm(sYcm); sSep = " :: "; 
        wrm(sSep); wrm("size=" + nXr + measUnit + "*" + nYr + measUnit); sSep = " :: "; 
        File fImg = new File(sLink);
        if(fImg.exists()) {              //-------------------- read it from the real exist image.
          BufferedImage myPicture = ImageIO.read(fImg);
          WritableRaster raster = myPicture.getRaster();
          int dX = raster.getWidth();
          int dY = raster.getHeight();
          wrm(sSep); wrm("px=" + dX + "*" + dY); sSep = " :: ";
          float dpiX = dX / nXinch, dpiY = dY / nYinch;
          int dpiXi = (int)(dpiX + 0.5f), dpiYi = (int)(dpiX + 0.5f);
          if(dpiXi == dpiYi ) {
            wrm(sSep); wrm("DPI = " + dpiXi);
          } else {
            wrm(sSep); wrm("DPI = " + dpiXi + "*" + dpiYi);
          }
        }
        wrAdoc("]\n");
        wrm(".>\n");               // write only one newline to combine it with the following paragraph text.
      }
    }    
    return null;
  }
  
  
  
  
  /**writes the appropriate ZmL for a &lt;text:a> which is a hyperlink in odg.
   * It regards a bookmark ref link.
   * @param ndA
   * @param iterNodes
   * @return
   * @throws IOException
   */
  private String gathWrXlink (XmlDataNode ndA, ListIterator<XmlDataNode> iterNodes, boolean bwrm) throws IOException {
    try {
      String sLink = ndA.getAttrib("xlink:href");
      if(sLink.contains("Impl-ReadOdg-PageShape-Pins")) Debugutil.stopp();
      String sText;
      if(ndA.text !=null) {
        sText = ndA.text;
      } else {
        sText = "";
        for(XmlDataNode ndText: ndA.allNodes) {              // get the link text.
          if(ndText.tag.equals("text:soft-page-break")) {
            sText += "\\|";                                  // optional break on this position.
          } else {
            String sText1 = ndText.getText();
            if(sText1 == null) {
              Debugutil.stop();
            } else {
              sText += sText1;                                 // gets also the text from a sub node as "text:span"
            }
          }
        }
      }
      int posLabel;
      //===================================================__ handle a link to the document suite __=======================================
      if( (posLabel = sLink.indexOf('#')) >=0              // Read the ref (bookmark) label as target in html
         && ( this.cmdArgs.sRefBesideHtml !=null && sLink.startsWith(this.cmdArgs.sRefBesideHtml)           // BBB
           || this.cmdArgs.sRefBesideInWWWHtml !=null && sLink.startsWith(this.cmdArgs.sRefBesideInWWWHtml)
           || this.cmdArgs.sRefBesidePdf !=null && sLink.startsWith(this.cmdArgs.sRefBesidePdf)
           || this.cmdArgs.sRefBesideInWWWPdf !=null && sLink.startsWith(this.cmdArgs.sRefBesideInWWWPdf) 
        )   ) {                //------------------------------ and the link starts with any of the given path 
        Debugutil.unexpected();  // it is excluded by test before.
        String sXref = sLink.substring(posLabel+1);        // for the files of this docu suite:
        if(sXref.equals("Impl-ReadOdg-PageShape-Pins")) Debugutil.stopp();
        sText = "";
        int nrNodes = 7;                                 // this is only if the Reference style is not found, do not skip over too much.
        boolean bReferenceText = false;
        //
        while(--nrNodes >=0 && iterNodes.hasNext()) { //---__ check the nodes after and gather the link text ______________--------------------
          XmlDataNode ndRef = iterNodes.next();
          if( ndRef.tag.equals("text:span")
                && ndRef.getAttrib("text:style-name").equals("Reference")) {  // The hlink stuff goes till the reference text
            sText += getTextXml(ndRef, true);            // The text <:Reference:... is the relavant text as info in <:@ref:
            bReferenceText = true;                       // break the loop here.
          }
          else if(ndRef.tag.equals("text:soft-page-break")) {
            nrNodes +=1;                                   // ignore a soft page break between this sequence;
          }
          else {                                 //---------- especially free text between the links, or in error situation continue with text
            if(bReferenceText) {                           // other than 
              iterNodes.hasPrevious();
              iterNodes.previous();
              break;
            }
            if(ndRef.tag.equals("text:a")) {
              nrNodes +=1;                                   // do not count the pdf ref node.
            }
            else {
              String sNodeTxt = getTextXml(ndRef, true);
              if(sNodeTxt !=null) {
                boolean bUseTxt = false;
                for(int ix = 0; ix < sNodeTxt.length(); ++ix) {
                  if("( )/:,".indexOf(sNodeTxt.charAt(ix))<0) {
                    bUseTxt = true; break;                   // use the text only if other than '() /:' are contained, but then use complete.
                  }                                          // means, ignore all stuff (...): / between the links.
                }
                if(bUseTxt) {
                  sText += sNodeTxt;                         // build the text to show with the file.pdf file or file.html
                }
              }
            }
          }
        } // while gather text after link
        this.lastXref = sXref;
        wrAdoc("<<"); wrAdoc(sXref);  
        if(bwrm) { wrm("<:@ref:#"); wrm(sXref); } 
        if(sText !=null) {
          wrAdoc(","); 
          if(bwrm) { wrm(":"); }
          if(this.sRefChapter !=null && this.sRefBookmark.equals(sXref)) {
            wrAdoc(this.sRefChapter); wrAdoc(" ");
            if(bwrm) { wrm(this.sRefChapter); wrm(" "); }
          }
          CharSequence textZml = replaceToBackslashSubscription(sText);  
          wrAdoc(textZml);
          if(bwrm) { wrm(textZml); }                       // sText in ZmL
        }
        wrAdoc(">>"); if(bwrm) { wrm(".>"); }  
      }
      else if(this.cmdArgs.sRefBesidePdf !=null && sLink.startsWith(this.cmdArgs.sRefBesidePdf)) { // if it comes here, "html / " link was not before.
        Debugutil.stop();
      }
      //===================================================^^ handle a link to the document suite ^^=======================================
      //
      else {
        if(sLink.endsWith("WriteOdt.html"))
          Debugutil.stop();
        if(sLink.startsWith(this.cmdArgs.swwwRoot) && this.lastHlinkPathToReplaced !=null ) {
          int posSameAsLocal = sLink.indexOf(this.lastHlinkPathToReplaced);
          if(posSameAsLocal >=0) {
            sLink = sLink.substring(0, posSameAsLocal)+ "...";
          }
          this.lastHlinkPathToReplaced = null;
        }
        else if(sLink.startsWith("../")) {   //----------------------- ../relative/link has always one ../ as start  
          sLink = sLink.substring(1);   //             change to: ./relative/link obviously as relative link in Asciidoc
          int pos1 = StringFunctions.indexAfterAnyChar(sLink, 0, -1, "./");
          if(pos1 >0) {   //--------------------------------------- candidate for replacing the hlink path for swwwLink
            this.lastHlinkPathToReplaced = sLink.substring(pos1);// only if it is a local path starting with .
            //System.out.print("\n: "+ this.lastHlinkPathToReplaced);
          }
        } 
        //
        if(sText == null) { sText = ""; }
        else if(sText.equals(sLink)) { sText = ""; }         // write only <:@link:path.> instead <:@link:path::path.>
        //else if(sText.endsWith("(...)")) {   //================== It is an ellipse for the arguments, cut 
          int posHash = sLink.indexOf('#');
          if(posHash >0) {
            int posNameEnd = StringFunctions.indexAfterIdentifier(sLink, posHash+1, -1, null);
            if(posNameEnd < sLink.length() -2 ) { // arguments are following, not only "--":
             //&& !sLink.endsWith("?")  ) {        // a "...?" on end is a stored non unique anchor, preserve it in ZmL  
              // Note: change the name with "?" appended. If the operation is found and replace,
              // then the sLinkCmp contains it, should be same as sLink if the operation is unchanged,
              // but can differ if meanwhile the operation was changed in argument types.
              // But then the "?" is removed.
              String sHtml = sLink.substring(0, posHash);
              String sTarget = sLink.substring(posHash+1);
              //String sTargetCmp = searchForReplacingLinkOperationLabel(sLink + "?", posHash, this.cmdArgs.dirIn);
              String sTargetCmp = searchForReplacingLinkOperationLabel(sHtml, sTarget, this.cmdArgs.dirIn);
              if(!sTargetCmp.endsWith("?")) {     // if the "?" remains on end, the operation is not found.
                // if sLink has had a "...?" on end, it may be replaced if now the operation is unique.
                String parenthesis = "(...)";
                if(posNameEnd == sLink.length()-2) {       // without arguments
                  parenthesis = "()";                      // it is important to write () because of space in docu.
                }
                sLink = sLink.substring(0, posNameEnd) + parenthesis; // only if operation is found. 
              }
            }
          }
        
        wrAdoc("link:"); wrAdoc(sLink); wrAdoc("["); 
        if(bwrm) { 
          wrReplaceLastSpace('\n');
          //if(sLink.startsWith("https://vishia.org/LibreOffc/pdf/Impl-OFB_VishiaDiagrams.pdf")) Debugutil.stopp();
          wrm("<:@link:"); wrm(sLink); 
        }
        if(sText.indexOf('\u200b') >=0)
          Debugutil.stop();
        CharSequence sTextWr = replaceToBackslashSubscription(sText); // returns sText if no changes are done.
        if(sText.length()>0) {
          wrAdoc(sTextWr);
          if(bwrm) { wrm("::"); wrm(sTextWr); }            // <:@link:https:/the/link::link text.>
        }                                                  // <:@link:https:/the/link.>
        if(bwrm) { wrm(".>"); }  
        wrAdoc("]");
        //this.cReplaceNextSpaceCharWith = '\n';
      }
    } catch(Exception exc) {      //========================= should be used as template for all how to write unexpected Exceptions. 
      CharSequence sExc = ExcUtil.exceptionInfo("", exc, 0, 20);
      this.log.writef("\nException %s", sExc);
      sExc = ExcUtil.exceptionInfo("Exception: ", exc, 0, 2);
      if(bwrm) { wrm("<:E:"); wrm(sExc); wrm(".>"); }
    }
    return null;
  }

  
  
  
  /**Gathers a &lt;text:bookmark-ref ...>, writes in Adoc as &lt;&lt;#label>>.
   * @param nodeXref
   * @return a char ' ' or '@' due tu the text:reference-format="chapter" or "page"
   *   to decide what to do with the text following, ignore space and ", page"
   * @throws IOException
   */
  private char gathWrXRef (XmlDataNode nodeXref) throws IOException {
    //String sXref = getBookmark(nodeXref, "text:ref-name");
    String sXref = nodeXref.attribs.get("text:ref-name");
    if(sXref.equals("Impl-ReadOdg-PageShape-Pins")) Debugutil.stopp();
    String sFormat = nodeXref.getAttrib("text:reference-format");
    String sText = nodeXref.getText();
    if(sXref.startsWith("__RefNumPara"))
      Debugutil.stop();
    char whatisit = '\0';
    if(sXref !=null && sFormat !=null) {
      boolean bManualRef = sXref.startsWith("__Link_");
      if(bManualRef) {
        sXref = sXref.substring(7);
      }
      if(sFormat !=null && sFormat.equals("chapter")) {
        this.sRefChapter = sText;
        this.sRefBookmark = sXref;
        whatisit = ' ';
      }
      else if(sFormat.equals("text")) { //=================== write only the ref for text with the title
        if(!bManualRef) {               //------------------- A generated Xref label, replace it by the manual one.
          LabelRef labelRef = this.idxLabelRef.get(sXref);
          if(labelRef == null) {
            Debugutil.stop();
          } else {
            if(labelRef.idxLabel !=null) {
              for(String sLabel : labelRef.idxLabel.values()) {
                sXref = sLabel;          //------------- xchange sXref with the manual set label if is it #__RefNum...
                break;  // use the first, should be only one anyway.
            } }
          }
        }
        this.lastXref = sXref;
        wrAdoc("<<"); wrAdoc(sXref); wrm("<:@ref:#"); wrm(sXref); 
        if(sText !=null) {
          wrAdoc(","); wrm(":"); 
          if(this.sRefChapter !=null && this.sRefBookmark.equals(sXref)) {
            wrAdoc(this.sRefChapter); wrAdoc(" ");
            wrm(this.sRefChapter); wrm(" ");
         }
          wrAdoc(sText);
          wrm(sText);
        }
        wrAdoc(">>"); wrm(".>");  
        this.sRefChapter = null;  // used.
      }
      else if(sFormat.equals("page")) {
        wrAdoc("PDF@"); wrAdoc(sText);
        wrm("<:@page.>"); //wrm(sText); wrm(".>");
      }
      else {  //--------------------------------------------- ignore all other refs, will automatically created.
        Debugutil.stop();
      }
    }
    return whatisit;
  }
  
  

  /**With the given direct style name the based indirect style is gotten.
   * If the indirect style stored in {@link #idxIndirectStyle} has the same properties for page break, column break ...
   * then only the indirect style is returned. Properties of the direct style are ignored.
   * But if the direct style as definitively properties which are important, then the direct style is returned.
   * The name of the direct style is completed with the specific properties,
   * and the returned direct style is stored in {@link #idxIndirectStyle} as quasi indirect style. 
   * It helps to save space if the same properties are given for another direct style (from view of LibreOffice internally)
   * but based on the same indirect style.  
   * @param styleNameArg
   * @return
   */
  private StyleOdt getBaseStyle (String styleNameArg, char[] cStyleSpan) {
    if(styleNameArg.equals("T18"))
      Debugutil.stop();
    if(styleNameArg.equals("T26"))
      Debugutil.stop();
    String styleName = styleNameArg;
    boolean pgBreak = false, colBreak = false; 
    char cItalic='?', cBold='?', cPosition ='?';
    StyleOdt style = this.idxDirectToStyle.get(styleName);
    //StyleOdt stylep = style;
    while(style !=null) {    //------------------------------ search back all parent styles, the base style has parent==null
      pgBreak |= style.pgBreakBefore;
      colBreak |= style.colBreakBefore;
      if(cPosition =='?' && style.cPosition !='?') { cPosition = style.cPosition; }
      if(cItalic =='?' && style.cItalic !='?') { cItalic = style.cItalic; }
      if(cBold =='?' && style.cWeight !='?') { cBold = style.cWeight; }
      
      //style = stylep;
      styleName = style.parentName;                       // parent name from the last found direct style
      if(styleName == null) {
        style = null;                                     // this is for a common direct style as 'italic' etc.
      } else {
        style = this.idxDirectToStyle.get(styleName);// parent or null if parent is an indirect style
      }
    }
    if(style == null ) {
      if(cBold !='?' || cItalic !='?' || cPosition !='?') { // any also indirect style can be modified.
        String styleNameModif = (styleName == null ? "modif_" : styleName) + cItalic + cBold + cPosition; 
        //styleName = "modif_" + (cItalic !='?' ? cItalic : "")  + (cBold !='?' ? cBold : "");  // modification style
        style = this.idxIndirectStyle.get(styleNameModif);
        if(style == null && styleName !=null) {
          style = this.idxIndirectStyle.get(styleName);
        }
        else { 
          styleName = styleNameModif;                      // use the styleNameModif to create this new style. 
        }
      } else if(styleName !=null) {
        style = this.idxIndirectStyle.get(styleName);      // the last style in the derivation is usual null, or the associated indirect style already set.
      }
      if(style == null && styleName !=null) {
        style = new StyleOdt('\0', styleName, null, cItalic, cBold, cPosition, false, false, 0); // create the missing indirect style.
        this.idxIndirectStyle.put(styleName, style);
      }
    }
    if(style !=null && (style.pgBreakBefore != pgBreak || style.colBreakBefore != colBreak)) {  // if the pgBreak or colBreak situation is set other than in indirect style:
      // we need a style as quasi indirect style with pgBreak or colBreak set.
      String styleNameBreak = styleName + (pgBreak ? "Pg" : "") + (colBreak ? "Col" : "");
      style = this.idxIndirectStyle.get(styleNameBreak);   // search or create the derived style with pgBreak or 
      if(style == null) {                                  // with the original style name of the existing indirect style
        style = new StyleOdt('\0', styleNameBreak, null, '?', '?', '?', pgBreak, colBreak, 0); // but with this flags.
        this.idxIndirectStyle.put(styleNameBreak, style);       // and store with this specific name.
      }
    }
    if((cItalic!='?' || cBold!='?') && cStyleSpan !=null) {
      // Quotation (or Citiation?), Emphasis with 'i', 'b', Strong Emphasis if only 'b' 
      cStyleSpan[0] = cItalic == 'i' ? (cBold == 'b' ? 'E': 'Q') : (cBold == 'b' ? 'S' : 0);
    }
    return style;
  }
  
  
  
  private int getNumber ( String sNr, int defaultNr) {
    int n = defaultNr;
    if(sNr !=null) {
      try { n = Integer.parseInt(sNr); } catch(NumberFormatException exc) { 
        this.log.writeError("EXCEPTION gatherText, parse " + sNr, exc);
      }
    }
    return n;
  }
  
  
  
  
  
  
  private int getAttribNumber ( XmlDataNode node, String key, int defaultNr) {
    String sNr = node.attribs == null ? null : node.attribs.get(key);
    return getNumber(sNr, defaultNr);
  }
  
  
  /**Writes a given text from XML to ZmL.
     * If the text contains specific designations given in {@link WriteOdt#sTextReplace} such as non-breaking spaces or the backslash
     * then the adequate transcription is used. This is in the same kind also in the WriteOdg direction. 
     * @param text The text from XML
     * @return null
     * @throws IOException
     */
    private String wrText (String textArg) throws IOException {
  //    if(textArg.startsWith("There are some non or bad visible characters"))
  //      Debugutil.stop();
      int pos0 = 0;
      if(textArg.startsWith("\\t"))
        Debugutil.stop();
      CharSequence sText = replaceToBackslashSubscription(textArg);
      String text = sText.toString();
      int pos1 = 0;
      if(!this.bInCodeBlock) {  //----------------------------- do not replace formatting stuff inside a code block (paragraph style Code...)
        int ztext = text.length();
        String[] checkFound = null;
        do {
          pos1 = ztext;
          int pos2= ztext;
          for(String[] check : WriteOdt.sTextCtrlCommon) { // is this yet used?? TODO
            int posi1 = text.indexOf(check[0], pos0);        // check all strings to enclose in pass
            if(posi1 >=0 && posi1 < ztext-1) {               // more left side found
              //assert(false);        // remove it, because <: is replaced.
              if(check.length >3) {
                int posi2 = text.indexOf(check[1], posi1 +1);
                if(posi2 > 0) {                              // only then check its end
                  if(pos1 > posi1) {                         // left and right side found, for example <<#....>>
                    pos1 = posi1;                            // then remark it, 
                    pos2 = posi2 + check[1].length();        // but maybe overridden by the next but more left side stuff 
                    checkFound = check;
                  }                                          // for example pass:[<<#] ... >>
                }
              } else {
                pos1 = posi1;
                pos2 = posi1 + check[0].length();
                checkFound = check;
              }
            }
          } // for
          if(pos1 < ztext) {                                 // any found in pass
            if(pos1 > pos0) { wrm(text.substring(pos0, pos1)); }
            if(checkFound.length==3) {                       // replace string
              assert( (pos2 - pos1) == checkFound[1].length());
              wrm(checkFound[0]);
            } else {
              //int posp = text.indexOf('+', pos1);              // if inside is no '+' then use +...+
              //if(posp >=pos1 && posp < pos2) {
              //  wrm("\\"); wrm(text.substring(pos1, pos2)); // else use the long form pass:[...]
              //} else {
              //  wrm("+"); wrm(text.substring(pos1, pos2)); wrm("+");
              //}
              wrm(text.substring(pos1, pos2));
            }
            pos0 = pos2;
          }
        } while(pos1 < ztext);
        if(pos0 < ztext) {
          wrm(text.substring(pos0));
        }   
      } else {  // code block
        assert(this.bInCodeBlock);
        if(textArg.startsWith("...text"))
          Debugutil.stop();
        wrm(text);
      }
      return null;
    }


    
    
  private static CharSequence replaceToBackslashSubscription( CharSequence textArg) {
    StringBuilder sbText = textArg instanceof StringBuilder ? (StringBuilder) textArg : null;  // use a given StringBuilder.
    CharSequence sText = textArg;
//    if(StringFunctions.contains(sText, "The bullet point"))
//      Debugutil.stop();
    int pos1 = 0;
    int[] ixA = new int[1];
    while(pos1 >=0) {          //============================ Replace some specific character sequences as part of text for example <: to <\:, \ to \\
      pos1 = StringFunctions.indexOfAnyChar(sText, pos1, Integer.MAX_VALUE, WriteOdt.sTextReplaceChars, ixA);
      if(pos1 >=0) {                                       // The first char of sTextReplace is found, may be also a simple '.'
        int ixRepl = ixA[0];                               // found String Array for replacement
//        if(ixRepl==0)
//          Debugutil.stop();
        String[] repl = WriteOdt.sTextReplace[ixRepl];     // the complete info
        int zRepl0 = repl[0].length();                     // number of chars to replace, often 1
        boolean bShouldReplace = zRepl0 == 1;
        if(zRepl0 >1) {           //------------------------- check the following characteres too! 
          do {
            bShouldReplace = StringFunctions.compare(sText, pos1, repl[0], 0, zRepl0) ==0;
            if(!bShouldReplace) {
              repl = WriteOdt.sTextReplace[++ixRepl];    // try to check the next
            }                                            // the replacements which same first char should be one after another!
          } while(!bShouldReplace && repl[0].charAt(0) == sText.charAt(pos1));
        }
        if(bShouldReplace) {
          if(sbText == null) {
            sbText = new StringBuilder(textArg);
            sText = sbText;
          }
          String sRepl = repl[1];
          sbText.replace(pos1, pos1+zRepl0,  sRepl);
          pos1 += sRepl.length();    // continue after the replaced String, do not replace the replacing. 
        } else {
          pos1 +=1;
        }
      }
    }
    return sText;          // it is the StringBuilder if used, or textArg
  }
    
    
    
    

  /**Writes a given text from XML to Asciidoc, but with detection of pass:[...] parts.
   * If the text contains specific designations given in {@link WriteOdt#sTextCtrlCommon}
   * then either <code>+text+</code> or <code>pass:[text]</code> is translated to Asciidoc
   * with the adequate text parts. This can/should be given in the same kind in the WriteOdg direction. 
   * @param text The text from XML
   * @return null
   * @throws IOException
   */
  private String wrTextAdoc (String textArg) throws IOException {
    if(textArg.contains("search"))
      Debugutil.stop();
    if(!this.bInCodeBlock) {  //----------------------------- do not replace formatting stuff inside a code block (paragraph style Code...)
      int pos1 = 0;
      StringBuilder sbText = null;
      CharSequence sText = textArg;
      int[] ix = new int[1];
      while(pos1 >=0) {
        pos1 = StringFunctions.indexOfAnyChar(sText, pos1, Integer.MAX_VALUE, WriteOdt.sTextReplaceChars, ix);
        if(pos1 >=0) {
          if(sbText == null) {
            sbText = new StringBuilder(textArg);
            sText = sbText;
          }
          String sRepl = WriteOdt.sTextReplace[ix[0]][2];
          sbText.replace(pos1, pos1+1,  sRepl);
          pos1 += sRepl.length();    // continue after the replaced String, do not replace the replacing. 
        }
      }
      int pos0 = 0;
      String[] checkFound = null;
      String text = sText.toString();
      int ztext = text.length();
      do {
        pos1 = ztext;
        int pos2= ztext;
        for(String[] check : WriteOdt.sTextCtrlCommon) {
          int posi1 = text.indexOf(check[2], pos0);        // check all strings to enclose in pass
          if(posi1 >=0 && posi1 < ztext-1) {               // more left side found
            if(check.length >3) {
              int posi2 = text.indexOf(check[3], posi1 +1);
              if(posi2 > 0) {                              // only then check its end
                if(pos1 > posi1) {                         // left and right side found, for example <<#....>>
                  pos1 = posi1;                            // then remark it, 
                  pos2 = posi2 + check[3].length();        // but maybe overridden by the next but more left side stuff 
                  checkFound = check;
                }                                          // for example pass:[<<#] ... >>
              }
            } else {
              pos1 = posi1;
              pos2 = posi1 + check[2].length();
              checkFound = check;
            }
          }
        }
        if(pos1 < ztext) {                                 // any found in pass
          if(pos1 > pos0) { wrAdoc(text.substring(pos0, pos1)); }
          if(checkFound.length==3) {                       // replace string
            assert( (pos2 - pos1) == checkFound[1].length());
            wrAdoc(checkFound[2]);
          } else {
            int posp = text.indexOf('+', pos1);              // if inside is no '+' then use +...+
            if(posp >=pos1 && posp < pos2) {
              wrAdoc("pass:["); wrAdoc(text.substring(pos1, pos2)); wrAdoc("]"); // else use the long form pass:[...]
            } else {
              wrAdoc("+"); wrAdoc(text.substring(pos1, pos2)); wrAdoc("+");
            }
          }
          pos0 = pos2;
        }
      } while(pos1 < ztext);
      if(pos0 < ztext) {
        wrAdoc(text.substring(pos0));
      }   
    } else {  // code block
      wrAdoc(textArg);
    }
    return null;
  }



  private void wrReplaceLastSpace(char cNew) {
    int posLastSpace = this.sb.length()-1;
    if(posLastSpace >=0 && this.sb.charAt(posLastSpace) == ' ') {
      this.sb.setCharAt(posLastSpace, cNew);
    }
  }
  
  
  private void wrAdoc (CharSequence text) throws IOException {
    if(this.wra !=null) {
      if(this.sba.length() > 2000) {
        wrBufferToFile(this.sba, this.wra, null, false);
      }
      if(text == null) {
        this.sba.append("????");
      } else {
        int posOutputStart = this.sba.length();
        this.sba.append(text);
      }
    }
  }
  
  
  private void wrm (CharSequence text) throws IOException {
    if(this.wr !=null) {          // do nothing write if processes a read only __PART_RD_
//      if(StringFunctions.startsWith(text, "<:Code"))
//        Debugutil.stop();
      if(this.bInCodeBlock) {
        assert(this.sb.length()==0);
        this.wr.append(text);
        if(this.cmdArgs.bDebug && StringFunctions.indexOf(text, 0, -1, '\n')>=0) {
          this.wr.flush();
        }
      } else {
        if(this.sb.length() > 2000) {
          wrBufferToFile(this.sb, this.wr, null, false);
        }
        if(text == null) {
          this.sb.append("????");
        } else {
          int posOutputStart = this.sb.length();
          this.sb.append(text);
        }
      }
    }
  }
  
  
  static final String[] sNewlineCheck = { "\n", ". " };
  
  /**Used for search a separator position if {@link #sNewlineCheck} is not found. 
   * Also used to search forward the next possible separator position after "<:@xxx.> further text.
   */
  static final String[] sNewlineCheckBack = { " :: ", ", ", ": ", " and", " ", ". ", "\n" };
  
  
  
  /**Searches a possible text line break for a beautification shortened text write style.
   * 
   * @param sb1 The whole buffer
   * @param posStart focus from here.
   * @param posEndFocus spread span from start of the focus to detect also <:@....> as long span.
   * @param posEndTextline spread or span from start ot length of a normal sentence till ". " or alternativel till ", " etc.
   * @return
   */
  @SuppressWarnings("static-method") 
  private int searchLineBreak ( boolean[] bInsideRef
  , StringBuilder sb1, int posStart, int posEndFocus, int posEndTextSentence, int posEndTextline
  ) {
    final int posBreak;
    String test = sb1.subSequence(posStart, posEndFocus).toString();
    //                                                     // first search newline till \n or ". "
    int posNl = StringFunctions.indexOfAnyString(sb1, posStart, posEndTextSentence, sNewlineCheck, null, null);
    if(posNl <0) {      //--if ". " is not in focus,       // posNl on last ", " etc.
      posNl = StringFunctions.lastIndexOfAnyString(sb1, posStart, posEndTextline, sNewlineCheckBack, null);
      if(posNl <0) {                                       // a longer line without spaces, maybe a link, <:@ should be find.
        Debugutil.stop();
      }
    }
    int posRef = bInsideRef[0] ? posStart : StringFunctions.indexOf(sb1, posStart, posEndFocus, "<:@");
    bInsideRef[0] = false; // default, set true if :: detected as line end.
    int posRefStart = posRef;
    if(posRef >posStart && (posNl <0 || posRef < posNl)) { // <:@link before posNl, use it, but go back to a line-breakable position. 
      posRefStart = StringFunctions.lastIndexOfAnyString(sb1, posStart, posRef, sNewlineCheckBack, null);
      if(posRefStart <0) { posRefStart = posStart; }       // no line break possible, from start 
    }
    String test2 = posNl <=0 ? null : sb1.substring(posStart, posNl);
//    if( test.startsWith("(<:@link:http://www.vishia.org/fbg/docuSrcJava_FBcl/org/vishia/fbcl/readSource/Dataflow2Eventchain_FBrd.html#mapEvPrepUpdInQueue::"))
//      Debugutil.stop();
    //
    if(posRefStart >=0 && (posNl <0 || posRefStart < posNl)) {       // posRefStart is interesting
      if(sb1.substring(posRef, posRef+8).equals("<:@image"))
        Debugutil.stop();
      int posRefEnd = StringFunctions.indexOf(sb1, posRef, posEndFocus, ".>");  // span goes to any "<:@xxx.>
      int posRefEndline = posRefEnd <0 ? -1 : StringFunctions.indexOfAnyString(sb1, posRefEnd, posEndFocus, sNewlineCheckBack, null, null);
      if(posRefEndline > posEndFocus) {
        posRefEndline = posEndFocus;                       // limit it to focus.
      }
      if(posNl < posRefEndline) {
        posNl = -1;                                        // posNl inside <:@xxx::xxx.> is invalid.
      }
      if(posRefEndline <0 || posRefEndline > posEndTextSentence) {
        int posEndLineDesired = posRefEnd < posEndTextSentence ? posRefEnd : posEndTextline;  // search :: till posRefEnd or till the desired end of line
        posRefEndline = StringFunctions.lastIndexOf(sb1, posRef, posEndLineDesired, "::");  // span only till "<:@xxx:: from desired line length
        if(posRefEndline < 0) {
          posRefEndline = StringFunctions.indexOf(sb1, posRef, posRefEnd, "::");  // span only till "<:@xxx::
        }
        if(posRefEndline >0) {
          if(posRefEndline > posEndFocus) {
            posRefEndline = posEndFocus;                       // limit it to focus.
          } else {
            posRefEndline +=2;                               // end of line is now after "<:@xxx::"
          }
          bInsideRef[0] = true;
        } else {
          posRefEndline = posRefEnd+2;                     // break hard after ".>", no line end end no :: found.
        }
      }
      String test3 = sb1.substring(posRefStart, posRefEndline > posRefStart ? posRefEndline : posNl);
      //
      if(posRefStart == posStart) {          //-------------- "<:@..." immediately as new line start, or "(<:@..."
        if(posNl >= posRefEndline) {         //-------------- "<:@xxx.> is shorter then posNl
          posBreak = posNl;                                // then break with end of sentence, "<:@xxx.> and further." matches in a line.
        } else if(posRefEndline >0) {        //-------------- "<:@xxx::.>" is given, 
          posBreak = posRefEndline;                         // then the lines goes anyway till .>
        } else {
          this.log.writeWarning("WARNING line cannot break %s", sb1.subSequence(posStart, posEndFocus));
          posBreak = -1;                                   // no .> found in this line, do not break;
        }
      } else if(posRefEndline <0) {  //-------------------------- The .> from <:@xxx.> is outside of the focus, not found
        posBreak = posRefStart;                            // Then anyway break "<:xxx", it is not on start of line.
      } else if(posNl >0 && posRefEndline < posNl) {   //-------- The <:@xxx.> is completely before the found ". " or \n
        posBreak = posNl;                                  // then break after the "." or on given \n
      } else if(posNl <0) {                //---------------- ". " not found,   
        if( posRefEndline < posEndTextline) {   //---------------- but The <:@xxx.> is completely inside normal length
          posBreak = posRefEndline;                          // then break after the ".>" 
        } else if(posRefStart > posStart){
          posBreak = posRefStart;
        } else {
          posBreak = posRefEndline;  // maybe -1 then no break;
        }
      } else {             //------------------------------ found posNl was inside <:xxx>, hence search a new outside
        posNl = StringFunctions.indexOfAnyString(sb1, posRefEndline, posEndTextSentence, sNewlineCheck, null, null);
        if(posNl >0) {
          posBreak = posNl;                              // break after ". " or on \n after <:xxx>
        } else if(posRefStart > posStart){
          posBreak = posRefStart;
        } else {
          posBreak = posRefEndline;  // maybe -1 then no break;
        }
      }
      if(posRefEndline>= 0 && posNl >=0 && posRefEndline >= posNl) {
        if(posRefStart <= posStart) {
          posNl = posRefEndline;                          // <:xxx.> on start of line, hence next newline after .>
        } else {
          posNl = posRefStart;                             // <:xxx.> on the next newline
        }
      }
      
    } else if (posNl ==0) { //------------------------------ posNl without <:xxx>
      posBreak = posNl;
    } else if (posNl >=0) { //------------------------------ posNl without <:xxx>
      posBreak = posNl;
    } else {   //------------------------------------------ posNl not found
      assert(posRefStart <0);  // in program: <:xxx> not found in range.
      //                                                 // search line break after ", " etc.
      posNl = StringFunctions.lastIndexOfAnyString(sb1, posStart, posEndTextline, sNewlineCheckBack, null);
      while(posNl > posStart && sb1.charAt(posNl-1) == '\\') {  // escape before, do not break here
        posNl = StringFunctions.lastIndexOfAnyString(sb1, posStart, posNl-1, sNewlineCheckBack, null);
      }
      if(posNl >0) {
        posBreak = posNl;
      } else {
        posBreak = posEndTextline;  // line is not able to broke.
      }
    }
    if(posBreak > posEndFocus)
      Debugutil.stop();
    return posBreak;
  }
  
  
  static String[][] sCheckReplace = new String[ WriteOdt.sTextCtrlCommon.length 
                                              + WriteOdt.sTextReplace.length]
                                              [];
  static { 
    int ix=-1;
    for(String[] check : WriteOdt.sTextCtrlCommon) {
      sCheckReplace[++ix] = check;
    }
    for(String[] check : WriteOdt.sTextReplace) {
      sCheckReplace[++ix] = check;
    }
  }
  
  
  /**Writes the buffer content to the file and supplements necessary line breaks for Asciidoc:<ul>
   * <li>A line break is set always after a ". " whereby the space is replaced by the line break.
   * <li>If the resulting line is longer than TDODO then search backward to a ", " TODO
   * </ul>
   * @param bEnd true then writes all, false then prevent the last content without dot on end. 
   *   if false, further output comes which is regarded later.
   * @throws IOException
   */
  private void wrBufferToFile (StringBuilder sb1, Writer wr1, String[] s2, boolean bEnd) throws IOException {
    int posStart = 0;
    int pos9 = sb1.lastIndexOf("\n");    // prevent the next not finished line, for changes.
    boolean[] bInsideRef = new boolean[1];
    while(posStart < pos9) {
//      if(sb1.substring(posStart, pos9).contains("Traditional often images are positioned "))
//        Debugutil.stop();
      final int posBreak;
      int posEndFocus = posStart + this.cmdArgs.adocLineLengthFocus; // max line length if a ". " or \n is in focus.
      if(posEndFocus > pos9) { posEndFocus = pos9+1; } // Note: \n included in posEndSentence to have a chance to find it.
      int posEndSentence = posStart + this.cmdArgs.adocLineSentence; // max shorted line length in focus
      if(posEndSentence > pos9) { posEndSentence = pos9+1;}  // Note: \n included in posEndSentence to have a chance to find it.
      int posEndShorted = posStart + this.cmdArgs.adocLineLength; // max shorted line length in focus
      if(posEndShorted > pos9) { posEndShorted = pos9+1;}  // Note: \n included in posEndSentence to have a chance to find it.
      posBreak = searchLineBreak(bInsideRef, sb1, posStart, posEndFocus, posEndSentence, posEndShorted);
      if(posBreak >=0) {
        char c1 = sb1.charAt(posBreak);
        if(c1 == '\n') {}
        else if(c1 == ' ') {
          sb1.setCharAt(posBreak, '\n');                  // insert \n instead space 
        }
        else if(sb1.charAt(posBreak+1) == ' '){
         sb1.setCharAt(posBreak+1, '\n');                  // insert \n instead space 
        } 
        else {  //------------------------------------------- this is an exception situation, only if line is not able to broke on a space.
          sb1.insert(posBreak, '\n');                      // then insert an hard newline and hence a space in the text on re-conversion.
          pos9 +=1;
        }
        posStart = posBreak +1;
      } else {               // posBreak <=0
        posStart = pos9;     // finish while.
      }
    }
    final String sNextLineStub;
    if(bEnd) {
      sNextLineStub = "";
      int zSb = sb1.length();
      if(zSb >0 && sb1.charAt(zSb-1) != '\n') {  
        sb1.append("\n");             //----------------- bEnd: append a newline as last.
      }
    } else {              //--------------------------------- prevent content after last line break for next usage.
      sNextLineStub = sb1.subSequence(pos9+1, sb1.length()).toString();
      sb1.delete(pos9+1, sb1.length());
    }
    wr1.append(sb1);
    wr1.flush();
    if(s2 !=null) {
      s2[0] = sb1.toString();
      if(sNextLineStub.length()>0)
        Debugutil.stop();
    }
    sb1.setLength(0); sb1.append(sNextLineStub);   // start newly with stub of last line.
  }



  private void wrRep(CharSequence line) throws IOException {
    if(this.wrRep !=null) {
      this.wrRep.append(line).append("\n");
    }
  }
  
  
  
  private void writeLabelRef () {
    if(this.cmdArgs.dirLabel !=null) {
      String sFileLabel = this.cmdArgs.sNameLabel;
      int posWildcard = sFileLabel.indexOf('*');
      if(posWildcard >=0) {
        sFileLabel = sFileLabel.substring(0, posWildcard) + this.cmdArgs.sNameDoc + sFileLabel.substring(posWildcard +1);
        File fLabel = new File(this.cmdArgs.dirLabel, sFileLabel);
        Writer wrLabel = null;
        try {
          wrLabel = new OutputStreamWriter(new FileOutputStream(fLabel), "UTF-8");
          for(LabelRef labelRef: this.listLabelRef) {
            if(labelRef.sTitle == null && labelRef.sRefHeading !=null) {       // this is for a second label, no title
              LabelRef labelRef2 = this.idxLabelRef.get(labelRef.sRefHeading); // get the title from here
              labelRef.sTitle = labelRef2.sTitle;
              labelRef.sPage = labelRef2.sPage;
            }
            if(labelRef.sTitle == null && labelRef.sRefNumPara !=null) {       // second chance: this is for a second label, no title
              LabelRef labelRef2 = this.idxLabelRef.get(labelRef.sRefNumPara); // get the title from here
              labelRef.sTitle = labelRef2.sTitle;
              labelRef.sPage = labelRef2.sPage;
            }
            for(final String sLabel : labelRef.idxLabel.values()) {
              wrLabel.write('#');                        // the #label
              wrLabel.write(sLabel == null ? "?": sLabel);
              wrLabel.write(' ');                        // label till space  
              wrLabel.write(labelRef.sTitle ==null ? "?" : labelRef.sTitle);
              wrLabel.write(" @");                       // title till @ trimmed
              wrLabel.write(labelRef.sPage ==null ? "?" : labelRef.sPage);
              wrLabel.write('\n');
          } }
        } catch (IOException exc) {
          CharSequence sError = ExcUtil.exceptionInfo("\nERROR writing label.txt", exc, 0, 10);
          this.log.writeError(sError);
        }
        if(wrLabel !=null) { FileFunctions.close(wrLabel); }
      }
    }
  }
  
  
  
  
  private String wrClose() throws IOException {
    if(this.sb.length() > 0) {
      wrBufferToFile(this.sb, this.wr, null, true);
    }
    this.wr.close();
    if(this.wrFrame !=null) {
      this.wrFrame.close();
    }
    this.wr = null;
    if(this.sba.length() > 0) {
      wrBufferToFile(this.sba, this.wra, null, true);
    }
    if(this.wra !=null) {
      this.wra.close();
      this.wra = null;
    }
    if(this.wrRep !=null) {
      this.wrRep.close();
      this.wrRep = null;
    }
    return null;
    
  }


  static class LabelRef {
    
    /**This is the master of the table.*/
    String sRefHeading;
    
    
    /**Reference label used for links to numbered chapters.*/
    String sRefNumPara;
    
    Map<String, String> idxLabel;
    
    String sTitle;
    
    String sPage;

    public LabelRef( String refHeading, String refNumPara, List<String> listLabel, String sTitle, String sPage
    ) {
      this.sRefHeading = refHeading;
      this.sRefNumPara = refNumPara;
      this.idxLabel = new TreeMap<>();
      for(String sLabel : listLabel) {
        if(sLabel.equals("<:"))
          Debugutil.stop();
        this.idxLabel.put(sLabel, sLabel);  // sort alphabetic
      }
      this.sTitle = sTitle;
      this.sPage = sPage;
    }
    
    
  }
  
}
